package cz.fidentis.analyst.kdtree;

import java.util.concurrent.Callable;

/**
 * A functor. When instantiated, it can be gradually applied to multiple k-d trees.
 * It inspects the state of the tree one by one, and (cumulatevely) computes results.
 * <p>
 * Implement this interface whenever you want to define new algorithm over a k-d tree.
 * </p>
 * <p>
 * The visitor can be instatiated either as synchronous or asynchronous. 
 * The {@link cz.fidentis.analyst.kdtree.KdTreeVisitor#visit} inspection method
 * of a synchronous visitor is applied immediately. On the contrary, if the
 * visitor is asynchronous, then the method only stores the inpected
 * k-d tree for later concurrent inspection. To trigger the inspection,
 * a {@link java.util.concurrent.ExecutorService} has to be used calling
 * the {@link cz.fidentis.analyst.kdtree.KdTreeVisitor#call} method asynchronously
 * and then waiting for the results of all triggered visitors. 
 * </p>
 *  <p>
 * If the visitor is thread-safe, then a single instance of the visitor
 * can visit concurrenly (and assynchronously) multiple k-d trees. Otherwise, 
 * the parallel inspection is still possible, but a new instance of the visitor 
 * has to be used for each k-d tree.
 * </p>
 *
 * @author Daniel Schramm
 */
public abstract class KdTreeVisitor implements Callable<KdTreeVisitor> {
    
    private boolean asynchronous;
    
    /**
     * K-d tree stored temporarily for later cuncurrent inspection via 
     * the {@link cz.fidentis.analyst.kdtree.KdTreeVisitor#visit} method.
     */
    private KdTree kdTree;

    /**
     * Constructor.
     * 
     * @param asynchronous If {@code true}, then asynchronous visitor is created.
     * (the {@link cz.fidentis.analyst.kdtree.KdTreeVisitor#visit} is not executed 
     * immediately but postponed for concurrent asynchronous execution). 
     * Otherwise, the visitor is created as sequential 
     * (the {@link cz.fidentis.analyst.kdtree.KdTreeVisitor#visit} is executed immediately.)
     */
    public KdTreeVisitor(boolean asynchronous) {
        this.asynchronous = asynchronous;
    }
    
    /**
     * Returns {@code true} if the implementation is thread-safe and then
     * <b>a single visitor instance</b> can be applied to multiple KD trees simultaneously.
     * If the visitor is not thread-safe, the concurrent inspection 
     * via the {@link cz.fidentis.analyst.kdtree.KdTreeVisitor#visit}
     * is still possible, but with multiple visitor instances - one for each tree.
     * <p>
     * Thread-safe implementation means that any read or write from/to the visitor's 
     * state is implemented as {@code synchronized}.
     * </p>
     * 
     * @return {@code true} if the implementation is thread-safe.
     */
    public boolean isThreadSafe() {
        return true;
    }
    
    /**
     * Returns {@code true} if this visitor was created as asynchronous.
     * 
     * @return {@code true} if this visitor was created as asynchronous.
     */
    public boolean isAsynchronous() {
        return this.asynchronous;
    }
    
    /**
     * The inspection method to be implememted by specific visitors.
     * 
     * @param kdTree KD tree to be visited
     */
    protected abstract void visitKdTree(KdTree kdTree);
    
    /**
     * The main visitation method, either invoked immediately (if the visitor 
     * was created a synchronous) or postponed to later parallel execution 
     * (if the visitor was created a asynchronous).
     * 
     * @param kdTree KD tree to be visited
     */
    public void visit(KdTree kdTree) {
        if (asynchronous) {
            this.kdTree = kdTree;
        } else {
            visitKdTree(kdTree);
        }
    }
    
    @Override
    public KdTreeVisitor call() throws Exception {
        visitKdTree(kdTree);
        return this;
    }
}
