package cz.fidentis.analyst.procrustes;

import cz.fidentis.analyst.face.HumanFace;
import cz.fidentis.analyst.feature.FeaturePoint;
import cz.fidentis.analyst.icp.IcpTransformation;
import cz.fidentis.analyst.mesh.core.MeshPoint;
import cz.fidentis.analyst.procrustes.exceptions.ProcrustesAnalysisException;
import cz.fidentis.analyst.procrustes.utils.ProcrustesAnalysisUtils;
import org.ejml.data.Matrix;
import org.ejml.simple.SimpleMatrix;
import org.ejml.simple.SimpleSVD;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import javax.vecmath.Point3d;


/**
 * @author Jakub Kolman
 */
public class ProcrustesAnalysis {

    private ProcrustesAnalysisFaceModel faceModel1;
    private ProcrustesAnalysisFaceModel faceModel2;

    private Point3d modelCentroid1;
    private Point3d modelCentroid2;

    /**
     * Constructor
     *
     * @param humanFace1
     * @param humanFace2
     * @throws ProcrustesAnalysisException
     */
    public ProcrustesAnalysis(
            HumanFace humanFace1,
            HumanFace humanFace2) throws ProcrustesAnalysisException {

        ProcrustesAnalysisFaceModel model1 = new ProcrustesAnalysisFaceModel(humanFace1);
        ProcrustesAnalysisFaceModel model2 = new ProcrustesAnalysisFaceModel(humanFace2);

        if (model1.getFeaturePointsMap().values().size() != model2.getFeaturePointsMap().values().size()) {
            throw new ProcrustesAnalysisException("Lists of feature points do not have the same size");
        }

        if (!ProcrustesAnalysisUtils.checkFeaturePointsType(
                model1.getFeaturePointValues(), model2.getFeaturePointValues())) {
            throw new ProcrustesAnalysisException("Lists of feature points do not have the same feature point types");
        }

        this.faceModel1 = model1;
        this.faceModel2 = model2;

        this.modelCentroid1 = ProcrustesAnalysisUtils.findCentroidOfFeaturePoints(model1.getFeaturePointValues());
        this.modelCentroid2 = ProcrustesAnalysisUtils.findCentroidOfFeaturePoints(model2.getFeaturePointValues());
    }

    /**
     * Method called for analysis after creating initial data in constructor. This method causes superimposition
     * and rotation of the faces.
     */
    public void analyze() {
        this.superImpose();
        this.rotate();
    }

    /**
     * Imposes two face models (lists of feature points and vertices) over each other
     */
    private void superImpose() {
        centerToOrigin(this.faceModel1, this.modelCentroid1);
        centerToOrigin(this.faceModel2, this.modelCentroid2);
    }

    /**
     * Centers given face model to origin that is given centroid.
     * Moves all vertices and feature points by value of difference between vertex/feature point and centroid.
     *
     * @param faceModel
     * @param centroid
     */
    private void centerToOrigin(ProcrustesAnalysisFaceModel faceModel, Point3d centroid) {
        for (FeaturePoint fp: faceModel.getFeaturePointsMap().values()) {
            fp.getPosition().x -= centroid.x;
            fp.getPosition().y -= centroid.y;
            fp.getPosition().z -= centroid.z;
        }
        for (MeshPoint v : faceModel.getVertices()) {
            v.getPosition().x = v.getX() - centroid.x;
            v.getPosition().y = v.getY() - centroid.y;
            v.getPosition().z = v.getZ() - centroid.z;
        }
    }

    /**
     * By rotation of matrices solves orthogonal procrustes problem
     */
    private void rotate() {
        SimpleMatrix primaryMatrix = ProcrustesAnalysisUtils.createMatrixFromList(this.faceModel2.getFeaturePointValues());
        SimpleMatrix transposedMatrix = ProcrustesAnalysisUtils.createMatrixFromList(
                this.faceModel1.getFeaturePointValues()).transpose();

        SimpleMatrix svdMatrix = transposedMatrix.mult(primaryMatrix);
        SimpleSVD<SimpleMatrix> singularValueDecomposition = svdMatrix.svd();
        SimpleMatrix transposedU = singularValueDecomposition.getU().transpose();
        SimpleMatrix rotationMatrix = singularValueDecomposition.getV().mult(transposedU);
        primaryMatrix = primaryMatrix.mult(rotationMatrix);

        this.faceModel2.setFeaturePointsMap(
                ProcrustesAnalysisUtils.createFeaturePointMapFromMatrix(
                        primaryMatrix, this.faceModel2));

        rotateVertices(this.faceModel2.getVertices(), rotationMatrix);
    }


    /**
     * Rotates all vertices.
     *
     * For more details check out single vertex rotation {@link #rotateVertex}.
     *
     * @param vertices
     * @param matrix
     */
    // if rotated vertices are drawn immediately it is better to set them after rotating them all
    // so it would be drawn just once
    private void rotateVertices(List<MeshPoint> vertices, SimpleMatrix matrix) {
        if (vertices != null) {
            for (int i = 0; i < vertices.size(); i++) {
                rotateVertex(vertices.get(i), matrix);
            }
        }
    }

    /**
     * Rotates vertex v by simulating matrix multiplication with given matrix
     *
     * @param v
     * @param matrix
     */
    private static void rotateVertex(MeshPoint v, SimpleMatrix matrix) {
        double x = ((v.getX() * matrix.get(0, 0))
                + (v.getY() * matrix.get(1, 0))
                + (v.getZ() * matrix.get(2, 0)));
        double y = ((v.getX() * matrix.get(0, 1))
                + (v.getY() * matrix.get(1, 1))
                + (v.getZ() * matrix.get(2, 1)));
        double z = ((v.getX() * matrix.get(0, 2))
                + (v.getY() * matrix.get(1, 2))
                + (v.getZ() * matrix.get(2, 2)));
        v.getPosition().x = x;
        v.getPosition().y = y;
        v.getPosition().z = z;
    }

}
