package cz.fidentis.analyst.kdtree;

import cz.fidentis.analyst.mesh.core.MeshFacet;
import java.util.Collection;

import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.vecmath.Vector3d;

/**
 * A single node of a KD-tree. Sharing verticess across meshes is supported
 * (the node links all faces that share the same vertex).
 *
 * @author Maria Kocurekova
 */
public class KdNode {
    /**
     * Current depth in the kd-tree
     */
    private final int depth;
    
    /**
     * Mesh facets sharing the stored vertex
     */
    private Map<MeshFacet, Integer> facets = new HashMap<>();

    /**
     * KD-tree topology
     */
    private KdNode lesser = null;
    private KdNode greater = null;
    private KdNode parent;

    /**
     * Constructor of KdNode
     *
     * @param facet Mesh facet containing the mesh vertex. Must not be null
     * @param index The index under which the vertex is stored in the mesh facet.
     *              Must be &gt;= 0
     * @param depth Depth of the node in the kd-tree. Must be &gt;= 0
     * @param parent Parent node. Can be null
     * @throws IllegalArgumentException if some parameter is wrong
     */
    public KdNode(MeshFacet facet, int index, int depth, KdNode parent) {
        if (facet == null) {
            throw new IllegalArgumentException("facet");
        }
        if (index < 0) {
            throw new IllegalArgumentException("index");
        }
        if (depth < 0) {
            throw new IllegalArgumentException("depth");
        }
        this.facets.putIfAbsent(facet, index);
        this.depth = depth;
        this.parent = parent;
    }
    
    /**
     * Constructor of KdNode
     * 
     * @param facets Mesh facets containing the mesh vertex. Must not be null or empty
     * @param indices Indices under which the vertex is stored in the mesh facet.
     *              Must be &gt;= 0 and their unmber has to correspond to the number of facets.
     * @param depth Depth of the node in the kd-tree. Must be &gt;= 0
     * @param parent Parent node. Can be null
     * @throws IllegalArgumentException if some parameter is wrong
     */
    public KdNode(List<MeshFacet> facets, List<Integer> indices, int depth, KdNode parent) {
        if (facets == null || facets.isEmpty()) {
            throw new IllegalArgumentException("facets");
        }
        if (indices == null || indices.isEmpty()) {
            throw new IllegalArgumentException("indices");
        }
        if (depth < 0) {
            throw new IllegalArgumentException("depth");
        }
        if (facets.size() != indices.size()) {
            throw new IllegalArgumentException("The number of facets and indiecs mismatch");
        }
        for (int i = 0; i < facets.size(); i++) {
            this.facets.putIfAbsent(facets.get(i), indices.get(i));
        }
        this.depth = depth;
        this.parent = parent;
    }

    /**
     * Adds another mesh facet to the node. This facet has to share a vertex 
     * (location coordinates) with vertex/facets already stored in the node. 
     * Otherwise, the behaviour may be unexpected.
     *
     * @param facet Mesh facet which has a vertex with the same coordinates as already stored vertex
     * @param index The index under which the vertex is stored in the mesh facet
     */
    public void addFacet(MeshFacet facet, int index){
        this.facets.putIfAbsent(facet, index);
    }

    public int getDepth() {
        return depth;
    }

    public Vector3d get3dLocation() {
        Map.Entry<MeshFacet, Integer> entry = facets.entrySet().iterator().next();
        return entry.getKey().getVertex(entry.getValue()).getPosition();
    }

    /**
     * Tree traversal - go to the "lesser" child.
     * 
     * @return lesser node, null if current node is leaf
     */
    public KdNode getLesser() {
        return lesser;
    }

    /**
     * Tree traversal - go to the "grater" child.
     * 
     * @return greater node, null if current node is leaf
     */
    public KdNode getGreater() {
        return greater;
    }

    /**
     * Tree traversal - go to the parent node.
     * 
     * @return parent of this node, null if current node is root
     */
    public KdNode getParent() {
        return parent;
    }

    /**
     * Set lesser node.
     *
     * @param lesser Node to be set as lesser
     * @return current node
     */
    public KdNode setLesser(KdNode lesser) {
        this.lesser = lesser;
        return this;
    }

    /**
     * Set lesser node.
     *
     * @param greater Node to be set as greater
     * @return current node
     */
    public KdNode setGreater(KdNode greater) {
        this.greater = greater;
        return this;
    }

    /**
     * Returns a map of all mesh facets that share the stored vertex. 
     * Value in the map contains the index which the vertex is stored in the mesh facet.
     * 
     * @return Map of facets sharing the stored mesh vertex
     */
    public Map<MeshFacet, Integer> getFacets() {
        return Collections.unmodifiableMap(facets);
    }

    /**
     * Set parent node.
     *
     * @param parent Node to be set as parent
     * @return current node
     */
    public KdNode setParent(KdNode parent) {
        this.parent = parent;
        return this;
    }
    
    @Override
    public String toString() {
        String ret = "";
        for (int i = 0; i < depth; i++) {
            ret += " ";
        }
        ret += "Node position: " + get3dLocation() + ", num. facets:  " + facets.size();
        ret += System.lineSeparator();
        return ret;
    }

}
