package cz.fidentis.analyst.newgui.canvas;

import com.jogamp.opengl.util.FPSAnimator;
import cz.fidentis.analyst.newgui.scene.Camera;
import java.util.Timer;
import java.util.TimerTask;

/**
 * Animates rotation and zooming of the camera in GLCanvas 
 * when pressing and holding manipulation button in the canvas.
 * 
 * @author Natalia Bebjakova 
 * @author Radek Oslejsek
 */
public class RotationAnimator {
    
    /**
     * @author Radek Oslejsek
     */
    public enum Direction {
        ROTATE_LEFT,
        ROTATE_RIGHT,
        ROTATE_UP,
        ROTATE_DOWN,
        ZOOM_IN,
        ZOOM_OUT,
        NONE
    }
    
    private final Camera camera;
    
    /**
     * Frequency of the rotation or zoom animations
     */
    private static final int FPS = 60; 
    
    /**
     * Animator used when the rotation or zoom buttons are pressed and held
     */
    private final FPSAnimator animator;
    
    /*
     * Animatio timer
     */
    private long startClickTime = 0;
    private TimerTask task;
    private Timer timer;
    
    private Direction direction = Direction.NONE;
    
    /**
     * Constructor.
     * 
     * @param glCanvas OpenGL canvas
     * @param camera Camera
     * @throws IllegalArgumentException if some argument is missing
     */
    public RotationAnimator(Canvas canvas) {
        if (canvas == null) {
            throw new IllegalArgumentException("canvas is null");
        }
        this.camera = canvas.getScene().getCamera();
        this.animator = new FPSAnimator(canvas.getGLCanvas(), FPS, true);
    }
    
    /**
     * Starts the animation.
     * 
     * @param dir Animation direction
     * @throws UnsupportedOperationException if there is unfinished animation
     */
    public void startAnimation(Direction dir) {
        if (this.direction != Direction.NONE) {
            throw new UnsupportedOperationException(); // this should no happen
        }
        
        animator.start();
        timer = new Timer();
        startClickTime = System.currentTimeMillis();
        task = new TimerTask() {
            @Override
            public void run() {
                switch (dir) {
                    case ROTATE_LEFT:
                        camera.rotateLeft(2);
                        break;
                    case ROTATE_RIGHT:
                        camera.rotateRight(2);
                        break;
                    case ROTATE_UP:
                        camera.rotateUp(2);
                        break;
                    case ROTATE_DOWN:
                        camera.rotateDown(2);
                        break;
                    case ZOOM_IN:
                        camera.zoomIn(3);
                        break;
                    case ZOOM_OUT:
                        camera.zoomOut(3);
                        break;
                    default:
                        throw new UnsupportedOperationException();
                }
            }
        };
        timer.schedule(task, 500, 100);
        
        this.direction = dir;        
    }
    
    /**
     * Stops the animation.
     */
    public void stopAnimation() {
        timer.cancel();
        if ((System.currentTimeMillis() - startClickTime) < 500) {
            switch (direction) {
                case ROTATE_RIGHT:
                    camera.rotateRight(22.5);
                    break;
                case ROTATE_LEFT:
                    camera.rotateLeft(22.5);
                    break;
                case ROTATE_UP:
                    camera.rotateUp(22.5);
                    break;
                case ROTATE_DOWN:
                    camera.rotateDown(22.5);
                    break;
                case ZOOM_IN:
                    camera.zoomIn(30);
                    break;
                case ZOOM_OUT:
                    camera.zoomOut(30);
                    break;
                default:
                    throw new UnsupportedOperationException();
            }
        }
        startClickTime = 0;
        animator.stop();
        this.direction = Direction.NONE;
    }
}
