package cz.fidentis.analyst.registration;

import com.jogamp.opengl.GL2;
import cz.fidentis.analyst.feature.FeaturePoint;
import cz.fidentis.analyst.gui.canvas.Canvas;
import cz.fidentis.analyst.gui.scene.DrawableFace;
import cz.fidentis.analyst.gui.scene.DrawableMesh;
import cz.fidentis.analyst.mesh.core.MeshFacet;
import cz.fidentis.analyst.mesh.core.MeshPoint;
import cz.fidentis.analyst.visitors.mesh.BoundingBox;
import java.awt.Color;
import java.util.ArrayList;
import java.util.List;
import javax.vecmath.Point3d;
import javax.vecmath.Vector3d;

/**
 * Updates primary resp. secondary {@link DrawableMesh}.
 * 
 * @author Richard Pajersky
 */

public class PostRegistrationListener {
    
    /**
     * Transformation shift is higher 
     */
    public static final double HIGH_SHIFT_QUOTIENT = 1;
    
    /**
     * Transformation shift is lower 
     */
    public static final double LOW_SHIFT_QUOTIENT = 0.1;
    
    /**
     * Quotient which determines translation and scale amount 
     */
    private static final double CHANGE_QUOTIENT = 500d;
    
    /**
     * Color used as default color of primary face 
     */
    public static final Color DEFAULT_PRIMARY_COLOR = new Color(101, 119, 179);
    
    /**
     * Color used as default color of secondary face 
     */
    public static final Color DEFAULT_SECONDARY_COLOR = new Color(237, 217, 76);
    
    /**
     * Range of transparency 
     */
    public static final int TRANSPARENCY_RANGE = 10;
    
    /**
     * Canvas which contains faces
     */
    private final Canvas canvas;
    
    /**
     * Move amount on X-axis
     */
    private final double moveX;
    
    /**
     * Move amount on Y-axis
     */
    private final double moveY;
    
    /**
     * Move amount on Z-axis
     */
    private final double moveZ;
    
    /**
     * Scale amount in all axis
     */
    private final double scaleXYZ;
    
    /**
     * Primary face used for visualization
     */
    private final DrawableFace primaryFace;
    
    /**
     * Secondary face used for visualization and transformation
     */
    private final DrawableFace secondaryFace;
    
    /**
     * Transformation amount
     */
    private double moveModifier = LOW_SHIFT_QUOTIENT;
    
    /**
     * Threshold of feature points showing too far away
     */
    private double featurePointsThreshold = LOW_SHIFT_QUOTIENT;
    
    /**
     * Constructor
     * First initialize {@link #canvas} then 
     * {@link #primaryFace} and {@link #secondaryFace}
     * Then computes movement amounts based on 
     * {@link BoundingBox} of secondary face
     * 
     * @param canvas Canvas with two faces
     * @throws IllegalArgumentException if the canvas is {@code null}
     */
    public PostRegistrationListener(Canvas canvas) {
        if (canvas == null) {
            throw new IllegalArgumentException("canvas is null");
        }
        this.canvas = canvas;
        
        List<DrawableFace> drawables = new ArrayList<>(canvas.getScene().getDrawables());
        if (drawables.size() < 2) {
            throw new IllegalArgumentException("canvas doesn't contains at least 2 models");
        }
        this.primaryFace = drawables.get(0);
        this.secondaryFace = drawables.get(1);
        setDeafultColor();
        
        // set move amounts
        BoundingBox visitor = new BoundingBox();
        this.secondaryFace.getModel().compute(visitor);
        Point3d maxPoint = visitor.getBoundingBox().getMaxPoint();
        Point3d minPoint = visitor.getBoundingBox().getMinPoint();
        moveX = (maxPoint.x - minPoint.x) / CHANGE_QUOTIENT;
        moveY = (maxPoint.y - minPoint.y) / CHANGE_QUOTIENT;
        moveZ = (maxPoint.z - minPoint.z) / CHANGE_QUOTIENT;
        scaleXYZ = (visitor.getBoundingBox().getMaxDiag() / (10 * CHANGE_QUOTIENT));
    }
    
    /**
     * Calculates feature points which are too far away
     * and changes their color to red
     * otherwise set color to default
     */
    private void calculateFeaturePoints() {
        if (!primaryFace.isRenderFeaturePoints()) {
            return;
        }
        ArrayList<Color> color = (ArrayList)secondaryFace.getFeaturePointsColor();
        for (int i = 0; i < primaryFace.getFeaturePoints().size(); i++) {
            FeaturePoint primary = primaryFace.getFeaturePoints().get(i);
            FeaturePoint secondary = secondaryFace.getFeaturePoints().get(i);
            Point3d transformed = new Point3d(secondary.getX(), secondary.getY(), secondary.getZ());
            transformPoint(transformed);
            double distance = Math.sqrt(
                Math.pow(transformed.x - primary.getX(), 2) + 
                Math.pow(transformed.y - primary.getY(), 2) + 
                Math.pow(transformed.z - primary.getZ(), 2));
            if (distance > featurePointsThreshold) {
                color.set(i, Color.RED);
            } else {
                color.set(i, DEFAULT_SECONDARY_COLOR);
            }
        }
    }
    
    /**
     * Applies carried out transformations
     * first on {@link #secondaryFace}
     * and then on {@link #secondaryFace} feature points
     */
    public void transformFace() {
        for (MeshFacet transformedFacet : secondaryFace.getFacets()) {
            for (MeshPoint comparedPoint : transformedFacet.getVertices()) {
                transformPoint(comparedPoint.getPosition());
            }
        }
        for (int i = 0; i < secondaryFace.getFeaturePoints().size(); i++) {
            FeaturePoint point = secondaryFace.getFeaturePoints().get(i);
            Point3d transformed = new Point3d(point.getX(), point.getY(), point.getZ());
            transformPoint(transformed);
            point = new FeaturePoint(transformed.x, transformed.y, 
                    transformed.z, point.getFeaturePointType());
            secondaryFace.getFeaturePoints().set(i, point);
        }
    }
    
    /**
     * Transforms point based on transformation info from {@link #secondaryFace}
     * 
     * @param point Point to transform
     */
    public void transformPoint(Point3d point) {
        if (point == null) {
            throw new IllegalArgumentException("point is null");
        }

        Point3d newPoint = new Point3d(0, 0, 0);
        double quotient;

        // rotate around X
        quotient = Math.toRadians(secondaryFace.getRotation().x);
        if (!Double.isNaN(quotient)) {
            double cos = Math.cos(quotient);
            double sin = Math.sin(quotient);
            newPoint.y = point.y * cos - point.z * sin;
            newPoint.z = point.z * cos + point.y * sin;
            point.y = newPoint.y;
            point.z = newPoint.z;
        }

        // rotate around Y
        quotient = Math.toRadians(secondaryFace.getRotation().y);
        if (!Double.isNaN(quotient)) {
            double cos = Math.cos(quotient);
            double sin = Math.sin(quotient);
            newPoint.x = point.x * cos + point.z * sin;
            newPoint.z = point.z * cos - point.x * sin;
            point.x = newPoint.x;
            point.z = newPoint.z;
        }

        // rotate around Z
        quotient = Math.toRadians(secondaryFace.getRotation().z);
        if (!Double.isNaN(quotient)) {
            double cos = Math.cos(quotient);
            double sin = Math.sin(quotient);
            newPoint.x = point.x * cos - point.y * sin;
            newPoint.y = point.y * cos + point.x * sin;
            point.x = newPoint.x;
            point.y = newPoint.y;
        }

        // translate
        point.x += secondaryFace.getTranslation().x;
        point.y += secondaryFace.getTranslation().y;
        point.z += secondaryFace.getTranslation().z;

        // scale
        point.x *= 1 + secondaryFace.getScale().x;
        point.y *= 1 + secondaryFace.getScale().y;
        point.z *= 1 + secondaryFace.getScale().z;
    }
    
    /**
     * Sets {@link #primaryFace} to {@link #DEFAULT_PRIMARY_COLOR} and
     * {@link #secondaryFace} to {@link #DEFAULT_SECONDARY_COLOR}
     */
    public final void setDeafultColor() {
        primaryFace.setColor(DEFAULT_PRIMARY_COLOR);
        secondaryFace.setColor(DEFAULT_SECONDARY_COLOR);
        canvas.renderScene();
    } 
    
    /**
     * Sets the color of {@link #primaryFace}
     * @param color Color
     */
    public void setPrimaryColor(Color color) {
        primaryFace.setColor(color);
        canvas.renderScene();
    }

    /**
     * Sets the color of {@link #secondaryFace}
     * @param color Color
     */
    public void setSecondaryColor(Color color) {
        secondaryFace.setColor(color);
        canvas.renderScene();
    }
    
    /**
     * @return {@link Color} of {@link #primaryFace}
     */
    public Color getPrimaryColor() {
        return primaryFace.getColor();
    }

    /**
     * @return {@link Color} of {@link #secondaryFace}
     */
    public Color getSecondaryColor() {
        return secondaryFace.getColor();
    }
    
    /**
     * Sets the transparency of {@link #primaryFace} or {@link #secondaryFace}
     * based on the inputed value and {@link #TRANSPARENCY_RANGE}
     * @param value Value
     */
    public void setTransparency(int value) {
        
        if (value == TRANSPARENCY_RANGE) {
            setPrimaryTransparency(1);
            setSecondaryTransparency(1);
        }
        if (value < TRANSPARENCY_RANGE) {
            setPrimaryTransparency(value / 10f);
            setSecondaryTransparency(1);
        }
        if (value > TRANSPARENCY_RANGE) {
            setSecondaryTransparency((2 * TRANSPARENCY_RANGE - value) / 10f);
            setPrimaryTransparency(1);
        }
        canvas.renderScene();
    }
    
    /**
     * Sets transparency of {@link #primaryFace}
     * 
     * @param transparency Transparency value
     */
    public void setPrimaryTransparency(float transparency) {
        primaryFace.setTransparency(transparency);
    }

    /**
     * Sets transparency of {@link #secondaryFace}
     * 
     */
    public void setSecondaryTransparency(float transparency) {
        secondaryFace.setTransparency(transparency);
    }
    
    /**
     * @return Transparency of {@link #primaryFace}
     */
    public double getPrimaryTransparency() {
        return primaryFace.getTransparency();
    }

    /**
     * @return Transparency of {@link #secondaryFace}
     */
    public double getSecondaryTransparency() {
        return secondaryFace.getTransparency();
    }
    
    /**
     * Sets camera to show face from the front
     */
    public void setFrontFacing() {
        canvas.getCamera().initLocation();
        canvas.renderScene();
    }
    
    /**
     * Sets camera to show face from the side
     */
    public void setSideFacing() {
        canvas.getCamera().initLocation();
        canvas.getCamera().rotate(0, 90);
        canvas.renderScene();
    }
    
    /**
     * Resets translation of {@link #secondaryFace}
     */
    public void resetTranslation() {
        secondaryFace.setTranslation(new Vector3d(0, 0, 0));
        calculateFeaturePoints();
        canvas.renderScene();
    }
    
    /**
     * Resets rotation of {@link #secondaryFace}
     */
    public void resetRotation() {
        secondaryFace.setRotation(new Vector3d(0, 0, 0));
        calculateFeaturePoints();
        canvas.renderScene();
    }

    /**
     * Resets scale of {@link #secondaryFace}
     */
    public void resetScale() {
        secondaryFace.setScale(new Vector3d(0, 0, 0));
        calculateFeaturePoints();
        canvas.renderScene();
    }
    
    /**
     * Sets translation of {@link #secondaryFace} on X-axis
     * @param value Translation amount
     */
    public void setXTranslation(double value) {
        secondaryFace.getTranslation().x = value * moveX;
        calculateFeaturePoints();
        canvas.renderScene();
    }
    
    /**
     * Sets translation of {@link #secondaryFace} on Y-axis
     * @param value Translation amount
     */
    public void setYTranslation(double value) {
        secondaryFace.getTranslation().y = value * moveY;
        calculateFeaturePoints();
        canvas.renderScene();
    }
        
    /**
     * Sets translation of {@link #secondaryFace} on Z-axis
     * @param value Translation amount
     */
    public void setZTranslation(double value) {
        secondaryFace.getTranslation().z = value * moveZ;
        calculateFeaturePoints();
        canvas.renderScene();
    }
    
    /**
     * Sets rotation of {@link #secondaryFace} around X-axis
     * @param value Translation amount
     */
    public void setXRotation(double value) {
        secondaryFace.getRotation().x = value * moveX;
        calculateFeaturePoints();
        canvas.renderScene();
    }
    
    /**
     * Sets rotation of {@link #secondaryFace} around Y-axis
     * @param value Translation amount
     */
    public void setYRotation(double value) {
        secondaryFace.getRotation().y = value * moveY;
        calculateFeaturePoints();
        canvas.renderScene();
    }
    
    /**
     * Sets rotation of {@link #secondaryFace} around Z-axis
     * @param value Translation amount
     */    
    public void setZRotation(double value) {
        secondaryFace.getRotation().z = value * moveZ;
        calculateFeaturePoints();
        canvas.renderScene();
    }
    
    /**
     * Sets scale of {@link #secondaryFace}
     * @param value Translation amount
     */
    public void setScale(double value) {
        secondaryFace.getScale().x = value * scaleXYZ;
        secondaryFace.getScale().y = value * scaleXYZ;
        secondaryFace.getScale().z = value * scaleXYZ;
        calculateFeaturePoints();
        canvas.renderScene();
    }
    
    /**
     * Sets color of {@link #primaryFace} highlights to white
     */
    public void setPrimaryHighlights() {
        primaryFace.setHighlights(new Color(1, 1, 1));
        canvas.renderScene();
    }
    
    /**
     * Sets color of {@link #secondaryFace} highlights to white
     */
    public void setSecondaryHighlights() {
        secondaryFace.setHighlights(new Color(1, 1, 1));
        canvas.renderScene();
    }
    
    /**
     * Sets color of {@link #primaryFace} highlights to black
     */
    public void removePrimaryHighlights() {
        primaryFace.setHighlights(new Color(0, 0, 0));
        canvas.renderScene();
    }
    
    /**
     * Sets color of {@link #secondaryFace} highlights to black
     */
    public void removeSecondaryHighlights() {
        secondaryFace.setHighlights(new Color(0, 0, 0));
        canvas.renderScene();
    }
    
    /**
     * Sets {@link #primaryFace} to be rendered as lines
     */
    public void setPrimaryLines() {
        primaryFace.setRenderMode(GL2.GL_LINE);
        canvas.renderScene();
    }
    
    /**
     * Sets {@link #secondaryFace} to be rendered as lines
     */
    public void setSecondaryLines() {
        secondaryFace.setRenderMode(GL2.GL_LINE);
        canvas.renderScene();
    }
    
    /**
     * Sets {@link #primaryFace} to be rendered as points
     */
    public void setPrimaryPoints() {
        primaryFace.setRenderMode(GL2.GL_POINT);
        canvas.renderScene();
    }
    
    /**
     * Sets {@link #secondaryFace} to be rendered as points
     */
    public void setSecondaryPoints() {
        secondaryFace.setRenderMode(GL2.GL_POINT);
        canvas.renderScene();
    }
    
    /**
     * Sets {@link #primaryFace} to be rendered solid
     */
    public void setPrimaryFill() {
        primaryFace.setRenderMode(GL2.GL_FILL);
        canvas.renderScene();
    }
    
    /**
     * Sets {@link #secondaryFace} to be rendered solid
     */
    public void setSecondaryFill() {
        secondaryFace.setRenderMode(GL2.GL_FILL);
        canvas.renderScene();
    }
    
    /**
     * @return {@code true} if feature points are being rendered
     */
    public boolean areFeaturePointsActive() {
        return primaryFace.isRenderFeaturePoints();
    }
    
    /**
     * Sets feature points of {@link #primaryFace} and {@link #secondaryFace}
     * to given state
     * @param state State
     */
    public void setFeaturePointsActive(boolean state) {
        primaryFace.setRenderFeaturePoints(state);
        secondaryFace.setRenderFeaturePoints(state);
        calculateFeaturePoints();
        canvas.renderScene();
    }

    /**
     * @return Value of {@link #featurePointsThreshold}
     */
    public double getFeaturePointsThreshold() {
        return featurePointsThreshold;
    }

    /**
     * Sets {@link #featurePointsThreshold} to given value
     * @param featurePointsThreshold Threshold
     */
    public void setFeaturePointsThreshold(double featurePointsThreshold) {
        this.featurePointsThreshold = featurePointsThreshold;
        calculateFeaturePoints();
        canvas.renderScene();
    }

    /**
     * @return Value of {@link #moveModifier}
     */
    public double getMoveModifier() {
        return moveModifier;
    }

    /**
     * Sets {@link #moveModifier} to given value
     * @param moveModifier Modifier
     */
    public void setMoveModifier(double moveModifier) {
        this.moveModifier = moveModifier;
    }

    /**
     * @return {@link #canvas}
     */
    public Canvas getCanvas() {
        return canvas;
    }
    
    /**
     * @return {@code true} if back face is being shown
     */
    public boolean isShowBackfaceActive() {
        return primaryFace.isShowBackface();
    }
    
    /**
     * Sets backface to be rendered or not based on the given state
     * @param state State
     */
    public void setShowBackfaceActive(boolean state) {
        primaryFace.setShowBackface(state);
        secondaryFace.setShowBackface(state);
        canvas.renderScene();
    }
}
