package cz.fidentis.analyst.project;

import cz.fidentis.analyst.faceState.FaceStatePanel;
import java.awt.Graphics2D;
import java.awt.image.BufferedImage;
import java.awt.Image;
import java.awt.event.ActionListener;
import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;
import javax.imageio.ImageIO;
import javax.swing.ImageIcon;
import javax.swing.JComboBox;
import javax.swing.table.DefaultTableModel;
import org.openide.util.Exceptions;
import org.imgscalr.Scalr;

/**
 * List of faces TableModel
 * 
 * @author Matej Kovar
 */
public class ModelsTableModel extends DefaultTableModel {
    
    // Black silhouette inserted when face has no preview
    private final ImageIcon previewBasic = new ImageIcon(FaceStatePanel.class.getClassLoader().getResource("/" + "face32x32.png"));
    private final ImageIcon previewBasicGray = new ImageIcon(FaceStatePanel.class.getClassLoader().getResource("/" + "face_gray32x32.png"));
    
    // List of paths faces which are currently filtered
    private List<Path> pathsToFilteredFaces = new ArrayList<>();
    private Integer filteredRows = 0;
    
    
    /**
     * Constructor
     */
    public ModelsTableModel() {
        super(new Object[]{"", "Models", "Tasks", "Preview"}, 0);
        
    }

    private Class[] types = new Class [] {
        java.lang.Boolean.class, java.lang.String.class, JComboBox.class, ImageIcon.class
    };

    private boolean[] canEdit = new boolean [] {
        true, false, true, false
    };

    @Override
    public Class getColumnClass(int columnIndex) {
        return types [columnIndex];}

    @Override
    public boolean isCellEditable(int rowIndex, int columnIndex) {

        if (rowIndex >= getRowCount() - filteredRows) {
            return false;
        }
        return canEdit [columnIndex];
    }
    
    @Override
    public void setRowCount(int rowCount) {
        
        if (rowCount == 0) {
            pathsToFilteredFaces.clear();
            filteredRows = 0;
            
        }
        super.setRowCount(rowCount);
    }
    
    @Override
    public void removeRow(int row) {
        
        String faceName = this.getValueAt(row, 1).toString();
        
        pathsToFilteredFaces.removeIf(path -> {
            return path.toString().substring(path.toString()
                                .lastIndexOf(File.separatorChar) + 1, 
                                path.toString().lastIndexOf('.')).equals(faceName);
        });
        
        super.removeRow(row);
    }
    
    /**
     * (De)selects all editable rows based on value passed to this function
     * @param selection (de)selects all editable rows
     */
    public void setAllRowsSelection(boolean selection) {
        
        for (int row = 0; row < this.getRowCount() - filteredRows; row++) {
            
            this.setValueAt(selection, row, 0);
            
        }
    }
    
    /**
     * Inflates selection
     */
    public void inflateSelection() {
        
        for (int row = 0; row < this.getRowCount() - filteredRows; row++) {
            
            if (this.getValueAt(row, 0) == (Object) true) {
                this.setValueAt(false, row, 0);
            } else {
                this.setValueAt(true, row, 0);
            }
            
        }
    }
    
    public Class[] getTypes() {
        return types;}

    public void setTypes(Class[] types) {
        this.types = types;}

    public boolean[] getCanEdit() {
        return canEdit;}

    public void setCanEdit(boolean[] canEdit) {
        this.canEdit = canEdit;
    }
    
    /**
     * Converts a given Image into a BufferedImage
     *
     * @param img The Image to be converted
     * @return The converted BufferedImage
     */
    public static BufferedImage toBufferedImage(Image img) {
        
        if (img instanceof BufferedImage) {
            return (BufferedImage) img;
        }
        
        if (img == null) {
            return null;
        }

        // Create a buffered image with transparency
        BufferedImage bimage = new BufferedImage(img.getWidth(null), img.getHeight(null), BufferedImage.TYPE_INT_ARGB);

        // Draw the image on to the buffered image
        Graphics2D bGr = bimage.createGraphics();
        bGr.drawImage(img, 0, 0, null);
        bGr.dispose();

        // Return the buffered image
        return bimage;
    }
    
    /**
     * Disables row because it is being filtered, i.e. disables preview
     * @param row which is currently filtered
     */
    public void disableRow(int row, Path path) {
        if (path != null && !pathsToFilteredFaces.contains(path)) {
            
            Image iconImage = ((ImageIcon)this.getValueAt(row, 3)).getImage();
            Path previewPath = this.getPreviewPath(path);
            
            if (previewPath == null || !previewPath.toFile().exists()) {
                
                this.setValueAt(previewBasicGray, row, 3);
                
            } else {
                
                BufferedImage image = toBufferedImage(iconImage);
                
                // Scalr.OP_DARKER passed 5 times to grey preview by 50%
                image = Scalr.apply(image, Scalr.OP_DARKER, Scalr.OP_DARKER, 
                        Scalr.OP_DARKER, Scalr.OP_DARKER, Scalr.OP_DARKER);
                this.setValueAt(new ImageIcon(image), row, 3);
            }
            
            pathsToFilteredFaces.add(path);
        }
    }
    
    /**
     * Enables row because it is not longer filtered, i.e. enables preview
     * @param row which is not longer filtered
     */
    public void enableRow(int row, Path path) {
        if (path != null && pathsToFilteredFaces.contains(path)) {
             
            Path previewPath = this.getPreviewPath(path);
            ImageIcon preview = null;
            
            if (previewPath == null || !previewPath.toFile().exists()) {
                preview = previewBasic;
            } else {
                try {
                BufferedImage image = ImageIO.read(previewPath.toFile());

                // Scale image to fit into column
                BufferedImage scaledImage = Scalr.resize(image, 70, 55);
                preview = new ImageIcon(scaledImage);

                } catch (IOException ex) {
                    Exceptions.printStackTrace(ex);
                }
                
            }
            
            this.setValueAt(preview, row, 3);
            pathsToFilteredFaces.remove(path);
        }
    }
    
    /**
     * Moves row to the bottom of the list (to filtered rows)
     * @param row index of row which is about to be moved/filtered
     */
    public void filterRow(int row) {
        moveRow(row, row, getRowCount() - 1);
        filteredRows++;
    }
    
    /**
     * Moves filtered rows to the top of the list (unfilters rows)
     * @param totalRows amount of rows which are about to be moved to the top
     */
    public void unfilterRows(int totalRows) {
        if (totalRows != 0) {
            moveRow(getRowCount() - totalRows, getRowCount() - 1, 0);
            filteredRows -= totalRows;   
        }
        
    }
    
    /**
     * Gets path to preview image
     * @param path path to face
     * @return path to preview if preview is found, null otherwise
     */
    public Path getPreviewPath(Path path) {
        
        Path preview = Paths.get(path.toString().substring(0, path.toString().lastIndexOf(".")).concat("_preview.jpg"));
        return preview;
    }
    
    
    /**
     * Adds new task to combo-box 
     * @param faceName name of the face
     * @param nameOfTask name of task
     */
    public void addNewTask(String faceName, String nameOfTask) {
        
        for (int row = 0; row < this.getRowCount(); row++) {
            if (this.getValueAt(row, 1).toString() == null ? faceName == null : this.getValueAt(row, 1).toString().equals(faceName)) {
                
                JComboBox comboBox = ((JComboBox)this.getValueAt(row, 2));
                
                comboBox.addItem(nameOfTask);
                return;   
            }
        }
    }
    
    /**
     * Removes task from combo-box
     * @param faceName name of face
     * @param nameOfTask name of task
     */
    public void removeTask(String faceName, String nameOfTask) {
        
        for (int row = 0; row < this.getRowCount(); row++) {
            if (this.getValueAt(row, 1).toString() == null ? faceName == null : this.getValueAt(row, 1).toString().equals(faceName)) {
                
                JComboBox comboBox = ((JComboBox)this.getValueAt(row, 2));
                
                comboBox.removeItem(nameOfTask);
                return;
            }
        }
    }
    
    /**
     * Adds new row to model
     * @param name String name of the face
     * @param path path to face
     * @param taskSelected task
     */
    public void addRowWithName(String name, Path path, ActionListener taskSelected) {
        
        Path preview = this.getPreviewPath(path);
        int index = this.getRowCount() - filteredRows;
        
        JComboBox comboBox = new JComboBox();
        comboBox.addActionListener(taskSelected);
        comboBox.setFont(new java.awt.Font("Tahoma", 0, 14));
        
        if (preview == null || !preview.toFile().exists()) {
            insertRow(index, new Object[]{false, name, comboBox, previewBasic});
        } else {
            try {
                BufferedImage image = ImageIO.read(preview.toFile());

                // Scale image to fit into column
                BufferedImage scaledImage = Scalr.resize(image, 70, 55);
                         
                insertRow(index, new Object[]{false, name, comboBox, new ImageIcon(scaledImage)});
                
            } catch (IOException ex) {
                Exceptions.printStackTrace(ex);
            }    
        }
    }
}
