package cz.fidentis.analyst.gui;

import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JSlider;
import javax.swing.JSpinner;
import javax.swing.SpinnerModel;
import javax.swing.SpinnerNumberModel;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.geom.AffineTransform;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;


/**
 * @author Daniel Sokol
 * <p>
 * Class for computing histogram and modifing distances
 */

public class HistogramComponent extends JPanel {
    private final JSpinner minSpinner = new JSpinner();
    private final JSpinner maxSpinner = new JSpinner();
    List<Double> values;
    int intervalsCount = 20;
    Double minValue = 1.0;
    Double maxValue = 10.0;
    Double oneStep = 1.0;
    int maxFrequency = 1;
    Map<Double, Integer> frequencies = new TreeMap<>();
    DrawCanvas drawnHistogram = new DrawCanvas();

    public HistogramComponent() {
        setupComponents();
    }

    /**
     * Add swing components to frame
     */
    private void setupComponents() {
        setBackground(new Color(0, 174, 163));
        setLayout(new BoxLayout(this, BoxLayout.PAGE_AXIS));

        JSlider intervalSlider = new JSlider(JSlider.HORIZONTAL, 0, 200, intervalsCount);
        intervalSlider.setMajorTickSpacing(20);
        intervalSlider.setMinorTickSpacing(5);
        intervalSlider.setPaintTicks(true);
        intervalSlider.setPaintLabels(true);
        intervalSlider.setBackground(new Color(0, 174, 163));

        add(intervalSlider);

        drawnHistogram.setMinimumSize(new Dimension(getMinimumSize().width, 270));
        drawnHistogram.setMaximumSize(new Dimension(getMaximumSize().width, 270));
        drawnHistogram.setPreferredSize(new Dimension(getPreferredSize().width, 270));

        add(drawnHistogram);
        add(new JLabel("Select minimum value of distances"));
        add(minSpinner);
        add(new JLabel("Select maximum value of distances"));
        add(maxSpinner);

        JButton newButton = new JButton("Set and Compute");

        newButton.addActionListener(e -> {

            intervalsCount = intervalSlider.getValue();
            minValue = (Double) minSpinner.getValue();
            maxValue = (Double) maxSpinner.getValue();
            setSpinners();
            cutValues();
            computeFrequencies();
            drawnHistogram.repaint();

        });
        add(newButton);

    }

    /**
     * Set spinners to new min a max values
     */
    private void setSpinners() {

        SpinnerModel minModel = new SpinnerNumberModel((double) minValue,
                (double) minValue, (double) maxValue, (double) oneStep);
        minSpinner.setModel(minModel);

        SpinnerModel maxModel = new SpinnerNumberModel((double) maxValue,
                (double) minValue, (double) maxValue, (double) oneStep);
        maxSpinner.setModel(maxModel);
    }


    public List<Double> getValues() {
        return values;
    }

    /**
     * Set new distances and compute their frequencies
     *
     * @param values
     */
    public void setValues(List<Double> values) {
        this.values = values;
        setMinMax();
        computeFrequencies();
        drawnHistogram.repaint();
        setSpinners();
    }

    public void computeFrequencies() {

        oneStep = maxValue - minValue;
        if (maxValue < 0) {
            oneStep = Math.abs(maxValue) - Math.abs(minValue);
        }
        oneStep /= intervalsCount;
        //Remove old values in frequencies
        frequencies.clear();

        for (Double i = minValue; i < maxValue; i += oneStep) {
            int currFrequency = 0; //Number of values in current interval
            for (Double temp : values) {
                if (temp >= i && temp < i + oneStep) {
                    currFrequency++;
                }
            }
            if (currFrequency > maxFrequency) {
                maxFrequency = currFrequency;
            }
            //Add beginning of interval and number of values in it
            frequencies.put(i, currFrequency);
        }
    }

    private void setMinMax() {
        minValue = values.get(0);
        maxValue = values.get(0);
        for (Double temp : values) {
            if (temp < minValue) {
                minValue = temp;
            }
            if (temp > maxValue) {
                maxValue = temp;
            }
        }
    }

    /**
     * Cut of distances that are before minimum or after maximum distance selected by user
     */
    private void cutValues() {
        for (int i = 0; i < values.size(); ++i) {
            if (values.get(i) < minValue) {
                values.set(i, minValue);
            } else if (values.get(i) > maxValue) {
                values.set(i, maxValue);
            }
        }
    }

    public Double getMinValue() {
        return minValue;
    }

    public Double getMaxValue() {
        return maxValue;
    }

    /**
     * @author Daniel Sokol
     * <p>
     * Internal class for drawing histogram
     */
    private class DrawCanvas extends JPanel {
        @Override
        public void paintComponent(Graphics g) {
            super.paintComponent(g);
            setBackground(new Color(0, 174, 163));

            int place = 0;

            int moveStep = this.getWidth() / intervalsCount;
            double heightModificator = (double) this.getHeight() / maxFrequency;

            Graphics2D g2 = (Graphics2D) g;
            Font font = new Font("Arial Narrow", Font.BOLD, 12);

            AffineTransform affineTransform = new AffineTransform();
            affineTransform.rotate(Math.toRadians(90), 0, 0);
            g2.setFont(font.deriveFont(affineTransform));

            for (Map.Entry<Double, Integer> entry : frequencies.entrySet()) {

                g.setColor(Color.ORANGE);
                int height = (int) (entry.getValue() * heightModificator);
                g2.fillRect(place, 300 - height, moveStep - 2, height);

                g.setColor(Color.BLACK);
                String temp = String.format("%.4f", entry.getKey());
                g2.drawString(temp, place, this.getHeight() - 50);
                place += moveStep;
            }
        }
    }
}
