package cz.fidentis.analyst.symmetry;

import cz.fidentis.analyst.Logger;
import cz.fidentis.analyst.canvas.Canvas;
import cz.fidentis.analyst.core.ControlPanelAction;
import cz.fidentis.analyst.face.HumanFace;
import cz.fidentis.analyst.face.events.HumanFaceEvent;
import cz.fidentis.analyst.face.events.HumanFaceListener;
import cz.fidentis.analyst.face.events.MeshChangedEvent;
import cz.fidentis.analyst.face.events.SymmetryPlaneChangedEvent;
import cz.fidentis.analyst.feature.FeaturePoint;
import cz.fidentis.analyst.mesh.core.MeshFacet;
import cz.fidentis.analyst.mesh.core.MeshModel;
import cz.fidentis.analyst.scene.DrawableFace;
import cz.fidentis.analyst.visitors.mesh.HausdorffDistance;
import cz.fidentis.analyst.visitors.mesh.HausdorffDistance.Strategy;

import java.awt.event.ActionEvent;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.swing.JTabbedPane;
import javax.vecmath.Point3d;
import javax.vecmath.Vector3d;

/**
 * Action listener for the manipulation with the symmetry plane.
 * 
 * @author Radek Oslejsek
 */
public class SymmetryAction extends ControlPanelAction implements HumanFaceListener  {

    private final SymmetryPanel controlPanel;
    private final JTabbedPane topControlPanel;
    private int sPlane = 0; // 0 = none, 1 = from mesh, 2 = from FPs

    /**
     * Constructor.
     * 
     * @param canvas OpenGL canvas
     * @param topControlPanel Top component for placing control panels
     */
    public SymmetryAction(Canvas canvas, JTabbedPane topControlPanel) {
        super(canvas, topControlPanel);
        this.topControlPanel = topControlPanel;
        //this.controlPanel = new SymmetryPanel(this);
        this.controlPanel = new SymmetryPanel(this);
        
        controlPanel.setComputeFromFPs(
                (getPrimaryFeaturePoints() != null && !getPrimaryFeaturePoints().getFeaturePoints().isEmpty()) || 
                        (getSecondaryFeaturePoints() != null && !getSecondaryFeaturePoints().getFeaturePoints().isEmpty())
        );

        // Place control panel to the topControlPanel
        this.topControlPanel.addTab(controlPanel.getName(), controlPanel.getIcon(), controlPanel);
        this.topControlPanel.addChangeListener(e -> {
            // If the symmetry panel is focused...
            if (((JTabbedPane) e.getSource()).getSelectedComponent() instanceof SymmetryPanel) {
                getCanvas().getScene().setDefaultColors();
                getCanvas().getScene().showSymmetryPlane(getCanvas().getScene().getPrimaryFaceSlot(), true);
                getCanvas().getScene().showSymmetryPlane(getCanvas().getScene().getSecondaryFaceSlot(), true);
            } else {
                getCanvas().getScene().showSymmetryPlane(getCanvas().getScene().getPrimaryFaceSlot(), false);
                getCanvas().getScene().showSymmetryPlane(getCanvas().getScene().getSecondaryFaceSlot(), false);
            }
        });

        // Be informed about changes in faces perfomed by other GUI elements
        getPrimaryDrawableFace().getHumanFace().registerListener(this);
        if (getSecondaryDrawableFace() != null) {
            getSecondaryDrawableFace().getHumanFace().registerListener(this);
        }
    }
    
    @Override
    public void actionPerformed(ActionEvent ae) {
        String action = ae.getActionCommand();
        
        switch (action) {
            case SymmetryPanel.ACTION_COMMAND_RECOMPUTE_FROM_MESH: 
                sPlane = 1;
                recomputeFromMesh(getCanvas().getScene().getPrimaryFaceSlot());
                recomputeFromMesh(getCanvas().getScene().getSecondaryFaceSlot());
                break; 
            case SymmetryPanel.ACTION_COMMAND_COMPUTE_FROM_FPS: 
                sPlane = 2;
                recomputeFromFeaturePoints(getCanvas().getScene().getPrimaryFaceSlot());
                recomputeFromFeaturePoints(getCanvas().getScene().getSecondaryFaceSlot());
                break;
            default:
                // do nothing
        }
        renderScene();
    }
    
    @Override
    public void acceptEvent(HumanFaceEvent event) {
        if (event instanceof MeshChangedEvent) { // recompute symmetry plane, if necessary
            switch (sPlane) {
                case 0: // none
                    return;
                case 1: // from mesh
                    recomputeFromMesh(event.getFace().equals(getPrimaryDrawableFace().getHumanFace()) 
                            ? getCanvas().getScene().getPrimaryFaceSlot()
                            : getCanvas().getScene().getSecondaryFaceSlot());
                    break;
                case 2: // from FPs
                    recomputeFromFeaturePoints(event.getFace().equals(getPrimaryDrawableFace().getHumanFace())
                            ? getCanvas().getScene().getPrimaryFaceSlot() 
                            : getCanvas().getScene().getSecondaryFaceSlot());
                    break;
                default:
            }
        }
        if (event instanceof SymmetryPlaneChangedEvent) {
            updatePrecision(event.getFace());
        }
    }
    
    protected void recomputeFromMesh(int index) {
        HumanFace face = getCanvas().getHumanFace(index);
        if (face != null) {
            Logger log = Logger.measureTime();
            SymmetryEstimator estimator = new SymmetryEstimator(controlPanel.getSymmetryConfig());        
            face.getMeshModel().compute(estimator);
            face.setSymmetryPlane(estimator.getSymmetryPlane());
            log.printDuration("Symmetry plane estimation from mesh for " + face.getShortName());
            setDrawablePlane(face, index);
        }
    }
    
    protected void recomputeFromFeaturePoints(int index) {
        HumanFace face = getCanvas().getHumanFace(index);
        if (face == null) {
            return;
        }
        
        Map<String, Point3d> processFP = new HashMap<>();
        List<Plane> planes = new ArrayList<>();
        
        for (FeaturePoint fp: face.getFeaturePoints()) {
            String code = fp.getFeaturePointType().getCode();
            String pairCode;
            if (code.endsWith("_R")) {
                pairCode = code.replaceAll("_R", "_L");
            } else if (code.endsWith("_L")) {
                pairCode = code.replaceAll("_L", "_R");
            } else {
                continue;
            }
            
            if (!processFP.containsKey(pairCode)) {
                processFP.put(code, fp.getPosition());
                continue;
            }
            
            Vector3d normal = new Vector3d(processFP.remove(pairCode));
            Vector3d center = new Vector3d(normal);
                
            normal.sub(fp.getPosition());
            normal.normalize();
                    
            center.add(fp.getPosition());
            center.scale(0.5);
                    
            planes.add(new Plane(normal, normal.dot(center)));
        }
        
        face.setSymmetryPlane(new Plane(planes)); // creates an average plane
        setDrawablePlane(face, index);
    }
    
    protected void setDrawablePlane(HumanFace face, int index) {
        getCanvas().getScene().setDrawableSymmetryPlane(index, face);
        getCanvas().getScene().getDrawableSymmetryPlane(index).setTransparency(0.5f);
        getCanvas().getScene().getDrawableSymmetryPlane(index).setColor(
                (index == 0) ? DrawableFace.SKIN_COLOR_PRIMARY.darker() : DrawableFace.SKIN_COLOR_SECONDARY.darker()
        );
    }
    
    protected void updatePrecision(HumanFace face) {
        if (face == null) {
            return;
        }
        
        MeshModel clone = new MeshModel(face.getMeshModel());
        
        for (MeshFacet facet: clone.getFacets()) {
            facet.getVertices().parallelStream().forEach(v -> {
                v.getPosition().set(face.getSymmetryPlane().reflectOverPlane(v.getPosition()));
            });
        }
        
        face.computeKdTree(false);
        HausdorffDistance visitor = new HausdorffDistance(
                face.getKdTree(), 
                Strategy.POINT_TO_POINT_DISTANCE_ONLY, 
                false, 
                true,
                false
        );
        clone.compute(visitor);
        controlPanel.updateHausdorffDistanceStats(visitor.getStats(), getCanvas().getHumanFace(0) == face);
    }
}
