package cz.fidentis.analyst.symmetry;

import cz.fidentis.analyst.canvas.Canvas;
import cz.fidentis.analyst.core.ComboSliderDouble;
import cz.fidentis.analyst.core.ControlPanelAction;
import cz.fidentis.analyst.face.HumanFace;
import cz.fidentis.analyst.face.events.HumanFaceEvent;
import cz.fidentis.analyst.face.events.HumanFaceListener;
import cz.fidentis.analyst.face.events.SymmetryPlaneChangedEvent;
import cz.fidentis.analyst.mesh.core.MeshFacet;
import cz.fidentis.analyst.mesh.core.MeshFacetImpl;
import cz.fidentis.analyst.scene.DrawableCuttingPlane;
import cz.fidentis.analyst.visitors.mesh.BoundingBox.BBox;
import cz.fidentis.analyst.visitors.mesh.CrossSectionZigZag;
import org.openide.filesystems.FileChooserBuilder;

import javax.swing.JTabbedPane;
import javax.swing.filechooser.FileNameExtensionFilter;
import javax.vecmath.Point3d;
import java.awt.event.ActionEvent;
import java.io.File;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.List;
import java.util.stream.Collectors;
import javax.swing.JComboBox;
import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.vecmath.Vector3d;

/**
 * Action listener for the manipulation with the cutting plane.
 *
 * @author Dominik Racek
 * @author Radek Oslejsek
 */
public class ProfilesAction extends ControlPanelAction implements HumanFaceListener {
    
    public static final double FEATUE_POINTS_CLOSENESS = 1.5;

    /*
     * GUI elements
     */
    private final ProfilesPanel controlPanel;
    private final JTabbedPane topControlPanel;

    /*
     * Calculated profiles
     */
    private CrossSectionCurve primaryCurve;
    private CrossSectionCurve secondaryCurve;
    private CrossSectionCurve primaryMirrorCurve;
    private CrossSectionCurve secondaryMirrorCurve;

    private double lastSliderValue = 0.5;
    private boolean cuttingPlaneFromSymmetry;
    
    private int priCuttingPlaneIndex = -1;
    private int secCuttingPlaneIndex = -1;    
    
    /**
     * Constructor.
     *
     * @param canvas          OpenGL canvas
     * @param topControlPanel Top component for placing control panels
     */
    public ProfilesAction(Canvas canvas, JTabbedPane topControlPanel) {
        super(canvas, topControlPanel);
        this.topControlPanel = topControlPanel;
        
        computeOrthogonalCuttingPlanes(false);

        //recomputePrimaryProfile();

        if (getSecondaryDrawableFace() != null) {
            //recomputeSecondaryProfile();
            controlPanel = new ProfilesPanel(this, this.primaryCurve, this.secondaryCurve);
            controlPanel.getProfileRenderingPanel().setSecondaryMirrorSegments(this.secondaryMirrorCurve);
        } else {
            controlPanel = new ProfilesPanel(this, this.primaryCurve);
        }
        controlPanel.getProfileRenderingPanel().setPrimaryMirrorSegments(this.primaryMirrorCurve);

        // Place control panel to the topControlPanel
        this.topControlPanel.addTab(controlPanel.getName(), controlPanel.getIcon(), controlPanel);
        this.topControlPanel.addChangeListener(e -> {
            // If the symmetry panel is focused...
            if (((JTabbedPane) e.getSource()).getSelectedComponent() instanceof ProfilesPanel) {
                getCanvas().getScene().setDefaultColors();
                showCuttingPlanes(true, controlPanel.isMirrorCutsChecked());
                recomputeProfiles();
            } else {
                showCuttingPlanes(false, false);
            }
        });
        
        showCuttingPlanes(false, false);
        
        // Be informed about changes in faces perfomed by other GUI elements
        getPrimaryDrawableFace().getHumanFace().registerListener(this);
        if (getSecondaryDrawableFace() != null) {
            getSecondaryDrawableFace().getHumanFace().registerListener(this);
        }
    }

    @Override
    public void actionPerformed(ActionEvent ae) {
        String action = ae.getActionCommand();
        
        switch (action) {
            case ProfilesPanel.ACTION_COMMAND_EXPORT:
                exportProfile(this.primaryCurve, "Export primary face profile to file");
                if (controlPanel.isMirrorCutsChecked()) {
                    exportProfile(this.primaryMirrorCurve, "Export primary face mirror profile to file");
                }

                if (getSecondaryDrawableFace() != null) {
                    exportProfile(this.secondaryCurve, "Export secondary face profile to file");
                    if (controlPanel.isMirrorCutsChecked()) {
                        exportProfile(this.secondaryMirrorCurve, "Export secondary face mirror profile to file");
                    }
                }
                break;
            case ProfilesPanel.ACTION_OFFSET_CUTTING_PLANE:
                double auxVal = ((ComboSliderDouble) ae.getSource()).getValue();
                double value =  auxVal - lastSliderValue;
                double multiplier = -150;
                
                setCuttingPlaneOffset(priCuttingPlaneIndex, multiplier * value);

                if (getSecondaryDrawableFace() != null) {
                    setCuttingPlaneOffset(secCuttingPlaneIndex, multiplier * value);
                }
                recomputeProfiles();
                lastSliderValue = auxVal;
                break;
            case ProfilesPanel.ACTION_CHANGE_CUTTING_PLANE:
                String option = (String)((JComboBox) ae.getSource()).getSelectedItem();
                
                if (option.equals(ProfilesPanel.OPTION_SYMMETRY_PLANE)) {
                    if (getScene().getDrawableSymmetryPlane(0) == null) {
                       JOptionPane.showMessageDialog(
                               new JFrame(), 
                               "Compute a symmetry plane at the Symmetry tab first.", 
                               "Dialog", 
                               JOptionPane.INFORMATION_MESSAGE
                       );
                       controlPanel.setDefaultPlaneSelection();
                       return;
                    }
                    controlPanel.resetSliderSilently();
                    computeCuttingPlanesFromSymmetry();
                } else if (option.equals(ProfilesPanel.OPTION_VERTICAL_PLANE)) {
                    controlPanel.resetSliderSilently();
                    computeOrthogonalCuttingPlanes(false);
                }
                
                showCuttingPlanes(true, controlPanel.isMirrorCutsChecked());
                recomputeProfiles();
                break;
            case ProfilesPanel.ACTION_MIRROR_CUTS:
                controlPanel.getProfileRenderingPanel().setMirrorCuts(controlPanel.isMirrorCutsChecked());
                showCuttingPlanes(true, controlPanel.isMirrorCutsChecked());
                break;
            default:
                // do nothing
        }
        renderScene();
    }
    
    @Override
    public void acceptEvent(HumanFaceEvent event) {
        if (event instanceof SymmetryPlaneChangedEvent && cuttingPlaneFromSymmetry) {
            controlPanel.resetSliderSilently();
            computeCuttingPlanesFromSymmetry(); // recompute cutting planes
            //getCanvas().getScene().showCuttingPlanes(false, controlPanel.isMirrorCutsChecked());
        }
    }

    private void exportProfile(CrossSectionCurve curve, String title) {
        File file = new FileChooserBuilder(ProfilesAction.class)
                .setTitle(title)
                .setDefaultWorkingDirectory(new File(System.getProperty("user.home")))
                .setFilesOnly(true)
                .setFileFilter(new FileNameExtensionFilter("csv files (*.csv)", "csv"))
                .setAcceptAllFileFilterUsed(true)
                .showSaveDialog();

        if (file == null) {
            return;
        }

        // If chosen file exists, use the exact file path
        // If chosen file does not exist and does not have an extension, add it
        if (!file.exists()) {
            if (!file.getAbsolutePath().endsWith(".csv")) {
                file = new File(file.getAbsolutePath() + ".csv");
            }
        }

        try {
            file.createNewFile();

            PrintWriter writer = new PrintWriter(file.getAbsoluteFile(), "UTF-8");
            writer.println("N,X-Coordinate,Z-Coordinate");
            for (int i = 0; i < curve.getNumSegments(); i++) {
                for (int j = 0; j < curve.getSegmentSize(i); ++j) {
                    writer.println(((i+1)*(j+1)) + "," + curve.getSegment(j).get(j).x + "," + curve.getSegment(i).get(j).z);
                }
            }

            writer.close();
        } catch (IOException ex) {
            System.out.println("ERROR writing to a file: " + ex);
        }
    }

    private void recomputePrimaryProfile() {
        //Main profile
        CrossSectionZigZag cs = new CrossSectionZigZag(getDrawableCuttingPlane(priCuttingPlaneIndex).getPlane());
        getPrimaryDrawableFace().getModel().compute(cs);
        this.primaryCurve = cs.getCrossSectionCurve();

        //Mirror profile
        CrossSectionZigZag mcs = new CrossSectionZigZag(getDrawableCuttingPlane(priCuttingPlaneIndex).getMirrorPlane());
        getPrimaryDrawableFace().getModel().compute(mcs);
        this.primaryMirrorCurve = mcs.getCrossSectionCurve();
    }

    private void recomputeSecondaryProfile() {
        //Main profile
        CrossSectionZigZag cs = new CrossSectionZigZag(getDrawableCuttingPlane(secCuttingPlaneIndex).getPlane());
        getSecondaryDrawableFace().getModel().compute(cs);
        this.secondaryCurve = cs.getCrossSectionCurve();

        //Mirror profile
        CrossSectionZigZag mcs = new CrossSectionZigZag(getDrawableCuttingPlane(secCuttingPlaneIndex).getMirrorPlane());
        getSecondaryDrawableFace().getModel().compute(mcs);
        this.secondaryMirrorCurve = mcs.getCrossSectionCurve();
    }

    private void recomputeProfiles() {
        recomputePrimaryProfile();
        projectCloseFeaturePoints(
                primaryCurve, 
                getCanvas().getPrimaryFace(), 
                getDrawableCuttingPlane(priCuttingPlaneIndex).getPlane());
        projectCloseFeaturePoints(
                primaryMirrorCurve, 
                getCanvas().getPrimaryFace(), 
                getDrawableCuttingPlane(priCuttingPlaneIndex).getPlane());
        controlPanel.getProfileRenderingPanel().setPrimarySegments(this.primaryCurve);
        controlPanel.getProfileRenderingPanel().setPrimaryMirrorSegments(this.primaryMirrorCurve);

        if (getSecondaryDrawableFace() != null) {
            recomputeSecondaryProfile();
            projectCloseFeaturePoints(
                    secondaryCurve, 
                    getCanvas().getSecondaryFace(), 
                    getDrawableCuttingPlane(secCuttingPlaneIndex).getPlane());
            projectCloseFeaturePoints(
                    secondaryMirrorCurve, 
                    getCanvas().getSecondaryFace(), 
                    getDrawableCuttingPlane(secCuttingPlaneIndex).getPlane());
            controlPanel.getProfileRenderingPanel().setSecondarySegments(this.secondaryCurve);
            controlPanel.getProfileRenderingPanel().setSecondaryMirrorSegments(this.secondaryMirrorCurve);
        }
    }

    private void setCuttingPlaneOffset(int index, double value) {
        //Move cutting planes left and mirror planes right
        //If normal is negative, need to negate value
        if (getDrawableCuttingPlane(index).getPlane().getNormal().x < 0) {
            getDrawableCuttingPlane(index).shift(-value);
        } else {
            getDrawableCuttingPlane(index).shift(value);
        }
    }
    
    /**
     * Creates vertical or horizontal cutting and mirror planes from bounding box.
     * 
     * @param horizontal 
     */
    protected void computeOrthogonalCuttingPlanes(boolean horizontal) {
        DrawableCuttingPlane drPlane = getDrawableOrthogonalPlane(horizontal);
        if (priCuttingPlaneIndex == -1) {
            priCuttingPlaneIndex = getCanvas().getScene().getFreeSlotForOtherDrawables();
        }
        getCanvas().getScene().setOtherDrawable(priCuttingPlaneIndex, drPlane);
        
        if (getSecondaryDrawableFace() != null) {
            if (secCuttingPlaneIndex == -1) {
                secCuttingPlaneIndex = getCanvas().getScene().getFreeSlotForOtherDrawables();
            }
            getCanvas().getScene().setOtherDrawable(secCuttingPlaneIndex, new DrawableCuttingPlane(drPlane));
        }
        cuttingPlaneFromSymmetry = false;
    }
    
    protected DrawableCuttingPlane getDrawableOrthogonalPlane(boolean horizontal) {
        BBox bbox = getCanvas().getPrimaryFace().getBoundingBox(); // use BBox of the first face for size and position estimation
        Plane plane = new Plane(
                (horizontal) ? new Vector3d(0,1,0) : new Vector3d(-1,0,0), 
                (horizontal) ? bbox.getMidPoint().y : bbox.getMidPoint().x
        );
        DrawableCuttingPlane cuttingPlane = new DrawableCuttingPlane(
                plane, 
                bbox.getMidPoint(), 
                bbox.getDiagonalLength(), 
                bbox.getDiagonalLength()
        );
        cuttingPlane.setTransparency(0.5f);
        return cuttingPlane;
    }
    
    /**
     * Copy rectangle from the symmetry plane.
     * 
     * @param faceIndex
     * @return 
     */
    protected DrawableCuttingPlane getCuttingPlaneFromSymmetry(int faceIndex) {
        MeshFacet symmetryFacet = getCanvas().getHumanFace(faceIndex).getSymmetryPlaneFacet();
        Plane symmetryPlane = getCanvas().getHumanFace(faceIndex).getSymmetryPlane();
        return new DrawableCuttingPlane(new MeshFacetImpl(symmetryFacet), new Plane(symmetryPlane));
    }
    
    /**
     * Creates cutting and mirror planes by copying the rectangle from the symmetry plane.
     */
    protected void computeCuttingPlanesFromSymmetry() {
        DrawableCuttingPlane cuttingPlane = getCuttingPlaneFromSymmetry(getCanvas().getPrimaryFaceIndex());
        cuttingPlane.setTransparency(0.5f);
        if (priCuttingPlaneIndex == -1) {
            priCuttingPlaneIndex = getCanvas().getScene().getFreeSlotForOtherDrawables();
        }
        getCanvas().getScene().setOtherDrawable(priCuttingPlaneIndex, cuttingPlane);
        recomputePrimaryProfile();
        
        if (getSecondaryDrawableFace() != null) {
            cuttingPlane = getCuttingPlaneFromSymmetry(getCanvas().getSecondaryFaceIndex());
            cuttingPlane.setTransparency(0.5f);
            if (secCuttingPlaneIndex == -1) {
                secCuttingPlaneIndex = getCanvas().getScene().getFreeSlotForOtherDrawables();
            }
            getCanvas().getScene().setOtherDrawable(secCuttingPlaneIndex, cuttingPlane);
            recomputeSecondaryProfile();
        }
        cuttingPlaneFromSymmetry = true;
    }

    protected void projectCloseFeaturePoints(CrossSectionCurve curve, HumanFace face, Plane cuttingPlane) {
        if (!face.hasFeaturePoints() || curve.getNumSegments() == 0) {
            return;
        }
        List<Point3d> close = face.getFeaturePoints()
                .stream()
                .filter(fp -> Math.abs(cuttingPlane.getPointDistance(fp.getPosition())) < FEATUE_POINTS_CLOSENESS)
                .map(fp -> fp.getPosition())
                .collect(Collectors.toList());
        curve.projectFeaturePointsToCurve(close);
    }
    
    protected DrawableCuttingPlane getDrawableCuttingPlane(int index) {
        return (DrawableCuttingPlane) getScene().getOtherDrawable(index);
    }
    
    protected void showCuttingPlanes(boolean show, boolean showMirrors) {
        if (getDrawableCuttingPlane(priCuttingPlaneIndex) != null) {
            getDrawableCuttingPlane(priCuttingPlaneIndex).show(show);
            getDrawableCuttingPlane(priCuttingPlaneIndex).showMirrorPlane(showMirrors);
        }
        if (getDrawableCuttingPlane(secCuttingPlaneIndex) != null) {
            getDrawableCuttingPlane(secCuttingPlaneIndex).show(show);
            getDrawableCuttingPlane(secCuttingPlaneIndex).showMirrorPlane(showMirrors);
        }
    }
}
