package cz.fidentis.analyst.canvas.toolbar;

import cz.fidentis.analyst.canvas.Canvas;
import cz.fidentis.analyst.scene.DrawableFace;
import java.awt.Color;
import java.awt.FlowLayout;
import java.awt.event.ActionEvent;
import javax.swing.AbstractAction;
import javax.swing.ImageIcon;
import javax.swing.JPanel;
import javax.swing.JSlider;
import javax.swing.JToggleButton;
import javax.swing.event.ChangeEvent;
import javax.swing.plaf.metal.MetalToggleButtonUI;
import org.openide.util.NbBundle;

/**
 * Canvas toolbox for scenes with a single faces. 
 * It enables to show-hide the face and its feature points,
 * and to change the transparency of the face.
 * 
 * @author Richard Pajersky
 * @author Radek Oslejsek
 */
public class SceneToolboxSingleFace extends JPanel {
    
    private static final String FACE_ICON = "head28x28.png";
    private static final String FEATURE_POINTS_ICON = "fps28x28.png";
    
    private static final int TRANSPARENCY_RANGE = 100;
    
    private JToggleButton primLandButton;
    private JToggleButton primFaceButton;
    private JSlider       slider;
    
    private final Canvas canvas;
    
    /**
     * Constructor.
     * @param canvas Rendering canvas
     */
    public SceneToolboxSingleFace(Canvas canvas) {
        this.canvas = canvas;
        initComponents();
    }

    private void initComponents() {
        JPanel panel = new JPanel();
        panel.setLayout(new FlowLayout());
        add(panel);
        
        primLandButton = new JToggleButton();
        primLandButton.setBorder(Canvas.BUTTONS_BORDER);
        primLandButton.setBackground(DrawableFace.SKIN_COLOR_PRIMARY);
        primLandButton.setUI(new MetalToggleButtonUI() {
            @Override
            protected Color getSelectColor() {
                return DrawableFace.SKIN_COLOR_PRIMARY.darker();
            }
        });
        primLandButton.setIcon(new ImageIcon(getClass().getResource("/" + FEATURE_POINTS_ICON)));
        primLandButton.setFocusable(false);
        primLandButton.setSelected(true);
        primLandButton.setToolTipText(NbBundle.getMessage(SceneToolboxSingleFace.class, "SceneToolboxSingleFace.landButton.tooltip"));
        panel.add(primLandButton);
        
        primLandButton.addActionListener(new AbstractAction() {
            @Override
            public void actionPerformed(ActionEvent e) {
                boolean onOff = ((JToggleButton) e.getSource()).isSelected();
                canvas.getScene().getDrawableFeaturePoints(canvas.getScene().getPrimaryFaceSlot()).show(onOff);
                canvas.renderScene();
            }
        });

        /////////////////////////
        primFaceButton = new JToggleButton();
        primFaceButton.setBorder(Canvas.BUTTONS_BORDER);
        primFaceButton.setBackground(DrawableFace.SKIN_COLOR_PRIMARY);
        primFaceButton.setUI(new MetalToggleButtonUI() {
            @Override
            protected Color getSelectColor() {
                return DrawableFace.SKIN_COLOR_PRIMARY.darker();
            }
        });
        primFaceButton.setIcon(new ImageIcon(getClass().getResource("/" + FACE_ICON)));
        primFaceButton.setFocusable(false);
        primFaceButton.setSelected(true);
        primFaceButton.setToolTipText(NbBundle.getMessage(SceneToolboxSingleFace.class, "SceneToolboxSingleFace.faceButton.tooltip"));
        panel.add(primFaceButton);
        
        primFaceButton.addActionListener(new AbstractAction() {
            @Override
            public void actionPerformed(ActionEvent e) {
                boolean onOff = ((JToggleButton) e.getSource()).isSelected();
                canvas.getScene().showDrawableFace(canvas.getScene().getPrimaryFaceSlot(), onOff);
                canvas.renderScene();
            }
        });
        
        /////////////////////////
        slider = new JSlider();
        //slider.setMajorTickSpacing(TRANSPARENCY_RANGE/2);
        slider.setMaximum(TRANSPARENCY_RANGE);
        //slider.setMinorTickSpacing(TRANSPARENCY_RANGE/2);
        slider.setPaintTicks(false);
        slider.setToolTipText(org.openide.util.NbBundle.getMessage(SceneToolboxSingleFace.class, "SceneToolboxSingleFace.slider.tooltip"));
        slider.setValue(TRANSPARENCY_RANGE);
        slider.setCursor(new java.awt.Cursor(java.awt.Cursor.HAND_CURSOR));
        panel.add(slider);
        
        slider.addChangeListener((ChangeEvent e) -> {
            int val = ((JSlider) e.getSource()).getValue();
            canvas.getScene().getDrawableFace(canvas.getScene().getPrimaryFaceSlot()).setTransparency(val/100f);
            canvas.renderScene();
        });
        
    }
}
