package cz.fidentis.analyst.canvas;

import com.jogamp.opengl.GLCapabilities;
import com.jogamp.opengl.GLProfile;
import com.jogamp.opengl.awt.GLCanvas;
import cz.fidentis.analyst.face.HumanFace;
import cz.fidentis.analyst.scene.Camera;
import cz.fidentis.analyst.scene.Scene;
import cz.fidentis.analyst.scene.SceneRenderer;
import cz.fidentis.analyst.canvas.toolbar.RenderingModeToolbox;
import cz.fidentis.analyst.scene.Drawable;
import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.event.ActionEvent;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.awt.image.BufferedImage;
import java.util.ArrayList;
import java.util.List;
import javax.swing.AbstractAction;
import javax.swing.ImageIcon;
import javax.swing.JButton;

import javax.swing.JLayeredPane;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JSeparator;
import javax.swing.JToggleButton;
import javax.swing.JToolBar;
import javax.swing.SwingConstants;
import javax.swing.border.Border;
import javax.swing.border.EmptyBorder;
import org.openide.awt.DropDownButtonFactory;
import org.openide.util.NbBundle;

/**
 * OpenGL canvas.
 * 
 * @author Natalia Bebjakova 
 * @author Radek Oslejsek
 */
public class Canvas extends JPanel {
    
    private static final String RENDERING_MODE_TOOLBOX_ICON = "wireframe28x28.png";
    private static final String BACKGROUND_BUTTON_ICON = "background28x28.png";
    private static final String REFLECTIONS_BUTTON_ICON = "reflections28x28.png";
    private static final Color  REFLECTIONS_COLOR = Color.DARK_GRAY;
    
    public static final Border BUTTONS_BORDER = new EmptyBorder(5,5,5,5);
    
    // Scene components:
    private final Scene scene = new Scene();
    private final SceneRenderer sceneRenderer;
    private final Camera camera = new Camera();
    private final List<HumanFace> faces = new ArrayList<>();
    
    // Listeners:
    private final CanvasListener listener;
    private final MouseRotationListener manipulator;
    
    // GUI elements:
    private JLayeredPane layeredPane;
    private JPanel canvasPanel;
    private ControlButtons controlButtonsPanel;
    private GLCanvas glCanvas;
    private JToolBar toolbar;
    
    /**
     * Constructor.
     */
    public Canvas() {
        sceneRenderer = new SceneRenderer();
        listener = new CanvasListener(this);
        manipulator = new MouseRotationListener(this);        
        initComponents();
    }
    
    /**
     * Adds a primary face to the scene and canvas.
     * 
     * @param face Face to added. 
     * @return index of the face in the scene, -1 on error.
     */
    public int addPrimaryFace(HumanFace face) {
        if (face != null) {
            faces.add(face);
            int index = scene.getFreeSlot();
            scene.setHumanFace(index, face);
            scene.setFaceAsPrimary(index);
            return index;
        }
        return -1;
    }
    
    /**
     * Adds a secondary face to the scene and canvas.
     * 
     * @param face Face to added. 
     * @return index of the face in the scene, -1 on error.
     */
    public int addSecondaryFace(HumanFace face) {
        if (face != null) {
            faces.add(face);
            int index = scene.getFreeSlot();
            scene.setHumanFace(index, face);
            scene.setFaceAsSecondary(index);
            return index;
        }
        return -1;
    }
    
    /**
     * Returns all faces in the scene.
     * 
     * @return all faces in the scene.
     */
    public List<HumanFace> getHumanFaces() {
        return this.faces;
    }
    
    /**
     * Returns the primary face or {@code null}
     * @return the primary face or {@code null}
     */
    public HumanFace getPrimaryFace() {
        return getHumanFace(getPrimaryFaceIndex());
    }
    
    /**
     * Returns the secondary face or {@code null}
     * @return the secondary face or {@code null}
     */
    public HumanFace getSecondaryFace() {
        return getHumanFace(getSecondaryFaceIndex());
    }

    /**
     * Returns face at the index or {@code null}
     * @param index Index of the face
     * @return human face or {@code null}
     */
    public HumanFace getHumanFace(int index) {
        return (index >= 0 && index < faces.size()) ? faces.get(index) : null;
    }
    
    /**
     * Returns index of the primary face or -1
     * @return index of the primary face or -1
     */
    public int getPrimaryFaceIndex() {
        return scene.getPrimaryFaceSlot();
    }
    
    /**
     * Returns index of the secondary face or -1
     * @return index of the secondary face or -1
     */
    public int getSecondaryFaceIndex() {
        return scene.getSecondaryFaceSlot();
    }
    
    /**
     * Returns the scene
     * @return the scene
     */
    public SceneRenderer getSceneRenderer() {
        return sceneRenderer;
    }
    
    /**
     * Renders the scene.
     */
    public void renderScene() {
        glCanvas.display();
    }
    
    /**
     * Returns the scene. 
     * 
     * @return the scene.
     */
    public Scene getScene() {
        return scene;
    }
    
    /**
     * Returns camera. 
     * 
     * @return camera
     */
    public Camera getCamera() {
        return camera;
    }
    
    /**
     * Returns the underlying OpenGL canvas.
     * 
     * @return the underlying OpenGL canvas.
     */
    public GLCanvas getGLCanvas() {
        return this.glCanvas;
    }
    
    /**
     * Sets background color
     * 
     * @param dark If {@code true}, then dark background is set
     */
    public void setDarkBackground(boolean dark) {
        if (dark) {
            canvasPanel.setBackground(SceneRenderer.DARK_BACKGROUND);
            sceneRenderer.setDarkBackground();
        } else {
            canvasPanel.setBackground(SceneRenderer.BRIGHT_BACKGROUND);
            sceneRenderer.setBrightBackground();
        }
    }
    
    /**
     * Adds a toolbox to the scene toolbar.
     * @param toolbox New toolbox
     */
    public void addToolBox(JPanel toolbox) {
        if (toolbox != null) {
            toolbar.add(toolbox);
        }
    }
    


    private ControlButtons getButtonsPanel(Canvas canvas) {
        ControlButtonsAction controlButtonsListener = new ControlButtonsAction(canvas);
        ControlButtons ret = new ControlButtons(controlButtonsListener);
        ret.setOpaque(false);
        ret.setBounds(20, 20, ControlButtons.SIZE.width, ControlButtons.SIZE.height);
        return ret;
    }
    
    private void initComponents() {
        setLayout(new BorderLayout());

        initGLCanvas();
        canvasPanel = new JPanel();
        canvasPanel.setLayout(new BorderLayout());
        canvasPanel.add(glCanvas);
        
        controlButtonsPanel = getButtonsPanel(this);
        
        layeredPane = new JLayeredPane();
        layeredPane.add(canvasPanel, 1);
        layeredPane.add(controlButtonsPanel, 0);
        add(layeredPane, BorderLayout.CENTER);
        
        setDarkBackground(true);
        
        toolbar = new JToolBar();
        toolbar.setFloatable(false);
        add(toolbar, BorderLayout.SOUTH);
        
        addComponentListener(new ComponentAdapter() {
            @Override
            public void componentResized(ComponentEvent e) {
                canvasPanel.setBounds(0, 0, layeredPane.getWidth(), layeredPane.getHeight());
                glCanvas.setBounds(layeredPane.getX(), layeredPane.getY(), layeredPane.getWidth(), layeredPane.getHeight());
            }
        });
        
        addToolbox(new RenderingModeToolbox(this), RENDERING_MODE_TOOLBOX_ICON, "Canvas.toolbar.renderingMode.tooltip");
        addBackgroundButton();
        addReflectionsButton();
        toolbar.add(new JSeparator(SwingConstants.VERTICAL));
        //addToolbox((couple) ? new SceneToolboxFaceToFace(this) : new SceneToolboxSingleFace(this),
        //        SCENE_TOOLBOX_BUTTON_ICON, "Canvas.toolbar.scene.tooltip");
    }

    private void initGLCanvas() {
        // gl version 2 is used
        GLCapabilities capabilities = new GLCapabilities(GLProfile.get(GLProfile.GL2));
        capabilities.setDoubleBuffered(true);
       
        // creates new glCanvas panel for displaying model
        glCanvas = new GLCanvas(capabilities);
        glCanvas.setVisible(true);     
        
        // enables glCanvas to react to events
        glCanvas.requestFocusInWindow();        
        //glCanvas.setSize(getWidth() - getInsets().left - getInsets().right, getHeight() - getInsets().top - getInsets().bottom);

        glCanvas.addGLEventListener(listener);
        glCanvas.addMouseListener(manipulator);
        glCanvas.addMouseMotionListener(manipulator);
        glCanvas.addMouseWheelListener(manipulator);        
    }
    
    private void addToolbox(JPopupMenu toolbox, String icon, String tooltip) {
        // The button that will display the default action and the
        // dropdown arrow
        JButton button = DropDownButtonFactory.createDropDownButton(
                new ImageIcon(new BufferedImage(16, 16, BufferedImage.TYPE_INT_ARGB)), 
                toolbox);
        
        button.setBorder(BUTTONS_BORDER);
        button.setIcon(new ImageIcon(getClass().getResource("/" + icon)));
        
        //Mnemonics.setLocalizedText(button, NbBundle.getMessage(RenderingToolBar.class, "RenderingToolBar.renderingmode.text"));
        button.setToolTipText(NbBundle.getMessage(Canvas.class, tooltip));
        button.setFocusable(false);
        //button.setText(null);
        toolbar.add(button);
    }
     
    private void addBackgroundButton() {
        JToggleButton button1 = new JToggleButton();
        button1.setBorder(BUTTONS_BORDER);
        button1.setIcon(new ImageIcon(getClass().getResource("/" + BACKGROUND_BUTTON_ICON)));
        button1.setFocusable(false);
        button1.setToolTipText(NbBundle.getMessage(Canvas.class, "Canvas.toolbar.background.tooltip"));
        
        button1.addActionListener(new AbstractAction() { 
            @Override
            public void actionPerformed(ActionEvent e) {
                setDarkBackground(! ((JToggleButton) e.getSource()).isSelected());
                renderScene();
            }
        });
        
        toolbar.add(button1);
        
    }
     
    private void addReflectionsButton() {
        JToggleButton button2 = new JToggleButton();
        button2.setBorder(BUTTONS_BORDER);
        button2.setIcon(new ImageIcon(getClass().getResource("/" + REFLECTIONS_BUTTON_ICON)));
        button2.setFocusable(false);
        button2.setToolTipText(NbBundle.getMessage(Canvas.class, "Canvas.toolbar.reflections.tooltip"));
        
        button2.addActionListener(new AbstractAction() { 
            @Override
            public void actionPerformed(ActionEvent e) {
                for (Drawable dm: getScene().getAllDrawables()) {
                    if (((JToggleButton) e.getSource()).isSelected()) {
                        dm.setHighlights(REFLECTIONS_COLOR);
                    } else {
                        dm.setHighlights(Color.BLACK);
                    }
                }
                renderScene();
            }
        });
        
        toolbar.add(button2);
    }
}
