package cz.fidentis.analyst.tests;

import cz.fidentis.analyst.face.HumanFace;
import cz.fidentis.analyst.icp.IcpTransformer;
import cz.fidentis.analyst.icp.NoUndersampling;
import cz.fidentis.analyst.visitors.mesh.HausdorffDistance;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.DoubleSummaryStatistics;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * A class for testing the efficiency of batch processing algorithms.
 * The goal of this tool is to measure time and precision of 
 * approximated Hausdorff distance (HD). 
 * 
 * - First, HD of the first face (primary face) with other faces is computed
 *   - Each secondary face is superimposed using ICP
 *   - Then it is stored in k-d tree 
 *   - Relative (!) HD from the primary to secondary face is computed and stored.
 *     The order cannot be reverted because we need HD matrices of the same size.
 *   - TODO: First face should be replaced with average face
 * - Then, the mutual distances are approximated:
 *   - For each pair of faces A and B, the distance is computed as the difference 
 *     between distances A-P and B-P, where P is the primary face.
 * 
 * @author Radek Oslejsek
 */
public class BatchSimilarityApproxHausdorff {
    private static final String DATA_DIR = "../../analyst-data-antropologie/_ECA";
    private static final String OUTPUT_FILE = "../../SIMILARITY_APPROX_HAUSDORFF.csv";
    
    /**
     * Main method 
     * @param args Input arguments 
     * @throws IOException on IO error
     */
    public static void main(String[] args) throws IOException, ClassNotFoundException, Exception {
        List<Path> faces = Files.list(new File(DATA_DIR).toPath())
                .filter(f -> f.toString().endsWith(".obj"))
                .sorted()
                .limit(100)
                .collect(Collectors.toList());
        
        
        Map<Integer, List<Double>> distances = new HashMap<>();
        
        for (int i = 1; i < faces.size(); i++) {
            HumanFace priFace = new HumanFace(faces.get(0).toFile());
            HumanFace secFace = new HumanFace(faces.get(i).toFile());
            
            secFace.computeKdTree(false);
            
            System.out.println(i + "/" + (faces.size()-1) + " (" + priFace.getShortName() + "/" + secFace.getShortName() + ")");
            
            IcpTransformer icp = new IcpTransformer(priFace.getMeshModel(), 50, true, 0.3, new NoUndersampling());
            secFace.getMeshModel().compute(icp, true);
            
            HausdorffDistance hd = new HausdorffDistance(secFace.getKdTree(), HausdorffDistance.Strategy.POINT_TO_POINT, true, true, true);
            priFace.getMeshModel().compute(hd, true);
            
            //System.out.println(hd.getDistances().get(priFace.getMeshModel().getFacets().get(0)).size());
            distances.put(i, hd.getDistances().get(priFace.getMeshModel().getFacets().get(0)));
        }
        
        BufferedWriter bfw = new BufferedWriter(new FileWriter(OUTPUT_FILE));
        bfw.write("PRI FACE;SEC FACE;MIN;MAX;AVG");
        bfw.newLine();
        
        for (int i = 0; i < faces.size(); i++) {
            for (int j = 0; j < faces.size(); j++) {
                if (i == j) {
                    continue;
                }
                
                DoubleSummaryStatistics stats;
                if (i == 0 || j == 0) {
                    stats = getStats(distances.get((i == 0) ? j : i));
                } else {
                    stats = getStats(distances.get(i), distances.get(j));
                }
                
                bfw.write(getShortName(faces.get(i).toFile())+";");
                bfw.write(getShortName(faces.get(j).toFile())+";");
                if (stats != null) {
                    bfw.write(String.format("%.20f", stats.getMin())+";");
                    bfw.write(String.format("%.20f", stats.getMax())+";");
                    bfw.write(String.format("%.20f", stats.getAverage())+"");
                    bfw.newLine();
                } else {
                    bfw.write(String.format("%.20f", Double.NaN)+";");
                    bfw.write(String.format("%.20f", Double.NaN)+";");
                    bfw.write(String.format("%.20f", Double.NaN)+"");
                    bfw.newLine();
                }
                bfw.flush();
            }
        }
        
        bfw.close(); 
        
    }    

    private static DoubleSummaryStatistics getStats(List<Double> d1, List<Double> d2) {
        if (d1.size() != d2.size()) {
            return null;
        }
        
        List<Double> ret = new ArrayList<>(d1.size());
        for (int i = 0; i < d1.size(); i++) {
            ret.add(Math.abs(d2.get(i) - d1.get(i)));
        }
        
        return ret.stream()
                .mapToDouble(Double::doubleValue)
                .summaryStatistics();
    }
    
    private static DoubleSummaryStatistics getStats(List<Double> d1) {
        List<Double> ret = new ArrayList<>(d1.size());
        for (int i = 0; i < d1.size(); i++) {
            ret.add(Math.abs(d1.get(i)));
        }
        
        return d1.stream()
                .mapToDouble(v -> Math.abs(v))
                .summaryStatistics();
    }
    
    private static String getShortName(File file) throws IOException {
        String id = file.getCanonicalPath();
        String name = id.substring(0, id.lastIndexOf('.')); // remove extention
        name = name.substring(id.lastIndexOf('/')+1, name.length());
        name = name.substring(name.lastIndexOf(File.separatorChar) + 1, name.length());
        return name;
    }
}
