package cz.fidentis.analyst.core;

import cz.fidentis.analyst.ProjectConfiguration;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.netbeans.api.settings.ConvertAsProperties;
import org.openide.awt.ActionID;
import org.openide.awt.ActionReference;
import org.openide.windows.TopComponent;
import org.openide.util.NbBundle.Messages;
import cz.fidentis.analyst.Project;
import cz.fidentis.analyst.face.HumanFace;
import cz.fidentis.analyst.dashboard.ModelsTableModel;
import cz.fidentis.analyst.dashboard.FilterPanel;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import javax.swing.AbstractAction;
import javax.swing.ImageIcon;
import javax.swing.JFileChooser;
import javax.swing.JOptionPane;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.filechooser.FileNameExtensionFilter;
import org.openide.filesystems.FileChooserBuilder;
import org.openide.util.Exceptions;

/**
 * The main panel enabling analysts to select the primary and secondary faces,
 * and to perform basic batch processing. This panel also serves as an entry
 * point for detailed face analysis and face-to-face comparison.
 *
 * @author Matej Kovar
 */
@ConvertAsProperties(
        dtd = "-//cz.fidentis.analyst.gui//Dashboard//EN",
        autostore = false
)
@TopComponent.Description(
        preferredID = "ProjectTopComp",
        //iconBase="SET/PATH/TO/ICON/HERE",
        persistenceType = TopComponent.PERSISTENCE_ALWAYS
)
@TopComponent.Registration(mode = "editor", openAtStartup = true)
@ActionID(category = "Window", id = "cz.fidentis.analyst.gui.ProjectTopComp")
@ActionReference(path = "Menu/Window" /*, position = 333 */)
@TopComponent.OpenActionRegistration(
        displayName = "#CTL_ProjectTopCompAction",
        preferredID = "ProjectTopComp"
)
@Messages({
    "CTL_ProjectTopCompAction=Project",
    "CTL_ProjectTopCompTopComponent=Project",
    "HINT_ProjectTopCompTopComponent=This is a Project window"
})
public final class ProjectTopComp extends TopComponent {

    private Project project;
    
    private List<FaceTab> tabs = new ArrayList<>();
    
    private ModelsTableModel model = new ModelsTableModel(new Object[]{"", "Models", "KD-tree"}, 0);
    
    private FilterPanel fp;
    
    /* List of indexes of selected Rows */
    private List<Integer> selectedRows = new ArrayList<>();
    
    private ObjectMapper mapper = new ObjectMapper();
    
    /**
     * Creates new ProjectTopComp, initializes new project
     */
    public ProjectTopComp() {
        project = new Project();
       
        initComponents();
        setName(Bundle.CTL_ProjectTopCompTopComponent());
        setToolTipText(Bundle.HINT_ProjectTopCompTopComponent());
        putClientProperty(TopComponent.PROP_CLOSING_DISABLED, Boolean.TRUE);
        putClientProperty(TopComponent.PROP_DRAGGING_DISABLED, Boolean.TRUE);
        putClientProperty(TopComponent.PROP_UNDOCKING_DISABLED, Boolean.TRUE);
        
        ActionListener listener = new AbstractAction() {
            @Override
            public void actionPerformed(ActionEvent e) {
                applyFilter();
            }
        };
        
        //fp = new FilterPanel(filterPanel, listener);
        
                
        // Execute infinite OutputWindowThread that redirects messages logged
        // via Logger into the standard output window
        OutputWindowThread.execute();

        openExistingOrNewProject();
        
    }

    /**
     * This method is called from within the constructor to initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is always
     * regenerated by the Form Editor.
     */
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {
        java.awt.GridBagConstraints gridBagConstraints;

        mainScrollPanel = new javax.swing.JScrollPane();
        mainPanel = new javax.swing.JPanel();
        buttonsPanel = new javax.swing.JPanel();
        addButton = new javax.swing.JButton();
        removeButton = new javax.swing.JButton();
        selectAllButton = new javax.swing.JButton();
        deselectAllButton = new javax.swing.JButton();
        inflateButton = new javax.swing.JButton();
        oneOnOneButton = new javax.swing.JButton();
        manyToManyButton = new javax.swing.JButton();
        analyseButton = new javax.swing.JButton();
        faceTableScrollPanel = new javax.swing.JScrollPane();
        table = new javax.swing.JTable();
        filterPanel = new javax.swing.JPanel();
        infoPanel = new javax.swing.JPanel();
        saveProjectButton = new javax.swing.JButton();
        openProjectButton = new javax.swing.JButton();
        newProjectButton = new javax.swing.JButton();

        setOpaque(true);

        buttonsPanel.setBorder(javax.swing.BorderFactory.createEtchedBorder());
        buttonsPanel.setMinimumSize(new java.awt.Dimension(0, 0));
        buttonsPanel.setLayout(new java.awt.GridBagLayout());

        addButton.setFont(new java.awt.Font("Tahoma", 0, 12)); // NOI18N
        org.openide.awt.Mnemonics.setLocalizedText(addButton, org.openide.util.NbBundle.getMessage(ProjectTopComp.class, "ProjectTopComp.addButton.text")); // NOI18N
        addButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                addButtonMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.ipadx = 20;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.insets = new java.awt.Insets(16, 16, 13, 4);
        buttonsPanel.add(addButton, gridBagConstraints);

        removeButton.setFont(new java.awt.Font("Tahoma", 0, 12)); // NOI18N
        org.openide.awt.Mnemonics.setLocalizedText(removeButton, org.openide.util.NbBundle.getMessage(ProjectTopComp.class, "ProjectTopComp.removeButton.text")); // NOI18N
        removeButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                removeButtonMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(16, 22, 13, 4);
        buttonsPanel.add(removeButton, gridBagConstraints);

        selectAllButton.setFont(new java.awt.Font("Tahoma", 0, 12)); // NOI18N
        org.openide.awt.Mnemonics.setLocalizedText(selectAllButton, org.openide.util.NbBundle.getMessage(ProjectTopComp.class, "ProjectTopComp.selectAllButton.text")); // NOI18N
        selectAllButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                selectAllButtonMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(16, 22, 13, 4);
        buttonsPanel.add(selectAllButton, gridBagConstraints);

        deselectAllButton.setFont(new java.awt.Font("Tahoma", 0, 12)); // NOI18N
        org.openide.awt.Mnemonics.setLocalizedText(deselectAllButton, org.openide.util.NbBundle.getMessage(ProjectTopComp.class, "ProjectTopComp.deselectAllButton.text")); // NOI18N
        deselectAllButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                deselectAllButtonMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 3;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(16, 22, 13, 4);
        buttonsPanel.add(deselectAllButton, gridBagConstraints);

        inflateButton.setFont(new java.awt.Font("Tahoma", 0, 12)); // NOI18N
        org.openide.awt.Mnemonics.setLocalizedText(inflateButton, org.openide.util.NbBundle.getMessage(ProjectTopComp.class, "ProjectTopComp.inflateButton.text")); // NOI18N
        inflateButton.setAlignmentX(0.5F);
        inflateButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                inflateButtonMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 4;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(16, 22, 13, 4);
        buttonsPanel.add(inflateButton, gridBagConstraints);

        oneOnOneButton.setFont(new java.awt.Font("Tahoma", 1, 12)); // NOI18N
        org.openide.awt.Mnemonics.setLocalizedText(oneOnOneButton, org.openide.util.NbBundle.getMessage(ProjectTopComp.class, "ProjectTopComp.oneOnOneButton.text")); // NOI18N
        oneOnOneButton.setAlignmentX(0.5F);
        oneOnOneButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                oneOnOneButtonMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 6;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(16, 11, 13, 4);
        buttonsPanel.add(oneOnOneButton, gridBagConstraints);

        manyToManyButton.setFont(new java.awt.Font("Tahoma", 1, 12)); // NOI18N
        org.openide.awt.Mnemonics.setLocalizedText(manyToManyButton, org.openide.util.NbBundle.getMessage(ProjectTopComp.class, "ProjectTopComp.manyToManyButton.text")); // NOI18N
        manyToManyButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                manyToManyButtonMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 5;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(16, 30, 13, 4);
        buttonsPanel.add(manyToManyButton, gridBagConstraints);

        analyseButton.setFont(new java.awt.Font("Tahoma", 1, 12)); // NOI18N
        org.openide.awt.Mnemonics.setLocalizedText(analyseButton, org.openide.util.NbBundle.getMessage(ProjectTopComp.class, "ProjectTopComp.analyseButton.text")); // NOI18N
        analyseButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                analyseButtonMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 7;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.insets = new java.awt.Insets(16, 11, 13, 4);
        buttonsPanel.add(analyseButton, gridBagConstraints);

        faceTableScrollPanel.setPreferredSize(new java.awt.Dimension(812, 750));

        table.setSize(faceTableScrollPanel.getWidth(), faceTableScrollPanel.getHeight());
        table.setFont(new java.awt.Font("Tahoma", 0, 18)); // NOI18N
        table.getTableHeader().setOpaque(false);
        table.getTableHeader().setBackground(new java.awt.Color(204,204,204));
        table.getTableHeader().setFont(new java.awt.Font("Tahoma", 0, 18));
        model.addTableModelListener(new TableModelListener() {

            public void tableChanged(TableModelEvent e) {
                jTable1TableChanged(e);
            }
        });
        table.setModel(model);
        table.getColumnModel().getColumn(0).setMaxWidth(50);
        table.getColumnModel().getColumn(2).setMaxWidth(75);
        table.getTableHeader().getColumnModel().getColumn(0).setMaxWidth(50);
        table.getTableHeader().getColumnModel().getColumn(2).setMaxWidth(75);
        table.setDragEnabled(true);
        table.setRowHeight(40);
        table.setSelectionBackground(new java.awt.Color(102, 204, 255));
        table.setSelectionMode(javax.swing.ListSelectionModel.SINGLE_SELECTION);
        table.getTableHeader().setReorderingAllowed(false);
        table.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                tableMouseClicked(evt);
            }
        });
        faceTableScrollPanel.setViewportView(table);

        filterPanel.setBorder(javax.swing.BorderFactory.createEtchedBorder());
        filterPanel.setMaximumSize(new java.awt.Dimension(342, 424));
        filterPanel.setPreferredSize(new java.awt.Dimension(342, 424));

        javax.swing.GroupLayout filterPanelLayout = new javax.swing.GroupLayout(filterPanel);
        filterPanel.setLayout(filterPanelLayout);
        filterPanelLayout.setHorizontalGroup(
            filterPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGap(0, 359, Short.MAX_VALUE)
        );
        filterPanelLayout.setVerticalGroup(
            filterPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGap(0, 420, Short.MAX_VALUE)
        );

        infoPanel.setBorder(javax.swing.BorderFactory.createEtchedBorder());
        infoPanel.setPreferredSize(new java.awt.Dimension(349, 363));

        javax.swing.GroupLayout infoPanelLayout = new javax.swing.GroupLayout(infoPanel);
        infoPanel.setLayout(infoPanelLayout);
        infoPanelLayout.setHorizontalGroup(
            infoPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGap(0, 0, Short.MAX_VALUE)
        );
        infoPanelLayout.setVerticalGroup(
            infoPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGap(0, 312, Short.MAX_VALUE)
        );

        infoPanel.setVisible(false);

        saveProjectButton.setFont(new java.awt.Font("Tahoma", 0, 12)); // NOI18N
        saveProjectButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/save100x24.png"))); // NOI18N
        org.openide.awt.Mnemonics.setLocalizedText(saveProjectButton, org.openide.util.NbBundle.getMessage(ProjectTopComp.class, "ProjectTopComp.saveProjectButton.text")); // NOI18N
        saveProjectButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                saveProjectButtonMouseClicked(evt);
            }
        });

        openProjectButton.setFont(new java.awt.Font("Tahoma", 0, 12)); // NOI18N
        openProjectButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/open100x24.png"))); // NOI18N
        org.openide.awt.Mnemonics.setLocalizedText(openProjectButton, org.openide.util.NbBundle.getMessage(ProjectTopComp.class, "ProjectTopComp.openProjectButton.text")); // NOI18N
        openProjectButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                openProjectButtonMouseClicked(evt);
            }
        });

        newProjectButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/new100x24.png"))); // NOI18N
        org.openide.awt.Mnemonics.setLocalizedText(newProjectButton, org.openide.util.NbBundle.getMessage(ProjectTopComp.class, "ProjectTopComp.newProjectButton.text")); // NOI18N
        newProjectButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                newProjectButtonMouseClicked(evt);
            }
        });

        javax.swing.GroupLayout mainPanelLayout = new javax.swing.GroupLayout(mainPanel);
        mainPanel.setLayout(mainPanelLayout);
        mainPanelLayout.setHorizontalGroup(
            mainPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(mainPanelLayout.createSequentialGroup()
                .addContainerGap()
                .addGroup(mainPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING, false)
                    .addComponent(buttonsPanel, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                    .addComponent(faceTableScrollPanel, javax.swing.GroupLayout.DEFAULT_SIZE, 863, Short.MAX_VALUE))
                .addGroup(mainPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.TRAILING)
                    .addGroup(mainPanelLayout.createSequentialGroup()
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED, 52, Short.MAX_VALUE)
                        .addComponent(newProjectButton)
                        .addGap(18, 18, 18)
                        .addComponent(saveProjectButton)
                        .addGap(18, 18, 18)
                        .addComponent(openProjectButton))
                    .addGroup(mainPanelLayout.createSequentialGroup()
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                        .addGroup(mainPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING, false)
                            .addComponent(filterPanel, javax.swing.GroupLayout.DEFAULT_SIZE, 363, Short.MAX_VALUE)
                            .addComponent(infoPanel, javax.swing.GroupLayout.DEFAULT_SIZE, 363, Short.MAX_VALUE))))
                .addGap(20, 20, 20))
        );
        mainPanelLayout.setVerticalGroup(
            mainPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(mainPanelLayout.createSequentialGroup()
                .addContainerGap()
                .addGroup(mainPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addGroup(mainPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                        .addGroup(mainPanelLayout.createSequentialGroup()
                            .addComponent(buttonsPanel, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                            .addGap(6, 6, 6))
                        .addGroup(javax.swing.GroupLayout.Alignment.TRAILING, mainPanelLayout.createSequentialGroup()
                            .addGroup(mainPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                                .addComponent(saveProjectButton)
                                .addComponent(openProjectButton))
                            .addGap(18, 18, 18)))
                    .addGroup(javax.swing.GroupLayout.Alignment.TRAILING, mainPanelLayout.createSequentialGroup()
                        .addComponent(newProjectButton)
                        .addGap(18, 18, 18)))
                .addGroup(mainPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING, false)
                    .addGroup(mainPanelLayout.createSequentialGroup()
                        .addComponent(filterPanel, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                        .addGap(28, 28, 28)
                        .addComponent(infoPanel, javax.swing.GroupLayout.PREFERRED_SIZE, 316, javax.swing.GroupLayout.PREFERRED_SIZE))
                    .addComponent(faceTableScrollPanel, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
                .addContainerGap())
        );

        saveProjectButton.getAccessibleContext().setAccessibleName(org.openide.util.NbBundle.getMessage(ProjectTopComp.class, "ProjectTopComp.saveProjectButton.AccessibleContext.accessibleName")); // NOI18N

        mainScrollPanel.setViewportView(mainPanel);

        mainScrollPanel.setSize(ControlPanel.CONTROL_PANEL_WIDTH, ControlPanel.HEIGHT);

        javax.swing.GroupLayout layout = new javax.swing.GroupLayout(this);
        this.setLayout(layout);
        layout.setHorizontalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addComponent(mainScrollPanel, javax.swing.GroupLayout.Alignment.TRAILING)
        );
        layout.setVerticalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addComponent(mainScrollPanel, javax.swing.GroupLayout.DEFAULT_SIZE, 871, Short.MAX_VALUE)
        );
    }// </editor-fold>//GEN-END:initComponents

    /**
     * Opens many to many tab
     * @param evt
     */
    private void manyToManyButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_manyToManyButtonMouseClicked
       createManyToManyTab("N:N");
    }//GEN-LAST:event_manyToManyButtonMouseClicked

    /**
     * Opens 1:1 tab with two selected faces, otherwise pops message that you
     * should select two faces
     * @param evt 
     */
    private void oneOnOneButton1MouseClicked(java.awt.event.MouseEvent evt) {                                             
        //loadTwoModels();

        if (selectedRows.size() == 2) {

            String name1 = model.getValueAt(selectedRows.get(0), 1).toString();
            String name2 = model.getValueAt(selectedRows.get(1), 1).toString();
            HumanFace face1 = project.getFaceByName(name1);
            HumanFace face2 = project.getFaceByName(name2);
            createFaceToFaceTab(face1, face2, name1 + ":" + name2, false);
        } else {
            JOptionPane.showMessageDialog(this, "Select two models");
        }
    }                                            

    /**
     * Inflates models (selected will be deselected and vice versa)
     * @param evt 
     */
    private void inflateButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_inflateButtonMouseClicked

        for (int i = 0; i < model.getRowCount(); i++) {
            if (model.getValueAt(i, 0) == (Object) true) {
                model.setValueAt(false, i, 0);
            } else {
                model.setValueAt(true, i, 0);
            }
        }
    }//GEN-LAST:event_inflateButtonMouseClicked

    /**
     * Deselects all models from list of models
     * @param evt 
     */
    private void deselectAllButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_deselectAllButtonMouseClicked
        deselectAllRows();
    }//GEN-LAST:event_deselectAllButtonMouseClicked

    /**
     * Deselects all rows
     * TODO : deselect only rows which are selected (checking row by row is slow)
     */
    private void deselectAllRows() {
        for (int i = 0; i < model.getRowCount(); i++) {
            model.setValueAt(false, i, 0);
        }
    }
    
    /**
     * Selects all models from list of models
     * @param evt 
     */
    private void selectAllButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_selectAllButtonMouseClicked
        for (int i = 0; i < model.getRowCount(); i++) {
            model.setValueAt(true, i, 0);
        }
    }//GEN-LAST:event_selectAllButtonMouseClicked

//GEN-FIRST:event_removeButton1MouseClicked
     /**
     * Removes selected models from list and project
     * @param evt Removes selected faces
     */
    private void removeButtonMouseClicked(java.awt.event.MouseEvent evt) {                                           
        removeSelectedFaces();
    }    
//GEN-LAST:event_removeButton1MouseClicked

    /**
     * Removes selected faces (those which are checked in check boxes)
     */
    private void removeSelectedFaces() {
        Collections.sort(selectedRows, Collections.reverseOrder());
        selectedRows.forEach(row -> {
            String name = model.getValueAt(row, 1).toString();
            List<FaceTab> tabsToClose = new ArrayList<>();
            tabs.stream().filter(t -> (t.hasFace(name))).forEachOrdered(t -> {
                tabsToClose.add(t);
            });
            
            while(!tabsToClose.isEmpty()) {
                tabsToClose.remove(0).close();
            }
            
            project.removeFaceByName(name);
            model.removeRow(row);
        });
        selectedRows.clear();        
        this.requestActive();
    }
    
    /**
     * Adds new model
     * @param evt 
     */
    private void addButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_addButtonMouseClicked
        loadModel(true);
    }//GEN-LAST:event_addButtonMouseClicked

    /**
     * Shows face state panel after clicking on face in list. Also double-click
     * on face opens Analyze tab for clicked face
     * @param evt 
     */
    private void tableMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_tableMouseClicked
        if (table.getSelectedRow() == -1) {
            infoPanel.setVisible(false);
        } else {
            
            if (evt.getClickCount() == 2) {
                deselectAllRows();
                model.setValueAt(true, table.getSelectedRow(), 0);
                analyseButtonMouseClicked(evt);
            }
            infoPanel.setVisible(true);
            checkFaceState(table.getValueAt(table.getSelectedRow(), 1).toString());       
        }
    }//GEN-LAST:event_tableMouseClicked

    /**
     * Saves current project
     * @param evt 
     */
    private void saveProjectButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_saveProjectButtonMouseClicked
        saveProject();
    }//GEN-LAST:event_saveProjectButtonMouseClicked

    /**
     * Open new project
     * @param evt 
     */
    private void openProjectButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_openProjectButtonMouseClicked
        if (loadNewProject()) {
            openProject();
        }
    }//GEN-LAST:event_openProjectButtonMouseClicked

    /**
     * Creates new project
     * @param evt 
     */
    private void newProjectButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_newProjectButtonMouseClicked
        if (loadNewProject()) {
            newProject();
        }
    }//GEN-LAST:event_newProjectButtonMouseClicked
    
    /**
     * Opens 1:1 tab with two selected faces, otherwise pops message that you
     * should select two faces
     * @param evt 
     */
    private void oneOnOneButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_oneOnOneButtonMouseClicked

        if (selectedRows.size() == 2) {

            String name1 = model.getValueAt(selectedRows.get(0), 1).toString();
            String name2 = model.getValueAt(selectedRows.get(1), 1).toString();

            HumanFace face1 = project.loadFace(name1);
            HumanFace face2 = project.loadFace(name2);
            
            if (project.getFaceByName(name1) == null) {
                face1.registerListener(model);
                project.addFace(face1);
            }
            
            if (project.getFaceByName(name2) == null) {
                face2.registerListener(model);
                project.addFace(face2);
            } 
            
            createFaceToFaceTab(face1, face2, name1 + ":" + name2, false);
        } else {
            JOptionPane.showMessageDialog(this, "Select two models");
        }
    }//GEN-LAST:event_oneOnOneButtonMouseClicked

    
    /**
     * Opens analysis of one selected face, otherwise pops message dialog that
     * you should select just one face
     * @param evt 
     */
    private void analyseButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_analyseButtonMouseClicked
        
        if (selectedRows.size() == 1) {

            String name = model.getValueAt(selectedRows.get(0), 1).toString();
            HumanFace face = project.loadFace(name);
            if (project.getFaceByName(name) == null) {
                face.registerListener(model);
                project.addFace(face);
            }
            
            createSingleFaceTab(face, name, false);
        } else {
            JOptionPane.showMessageDialog(this, "Select one model");
        }
    }//GEN-LAST:event_analyseButtonMouseClicked

    /**
     * Updates selectedRows - adds new selected rows or removes deselected rows
     * @param e TableModelEvent
     */
    private void jTable1TableChanged(TableModelEvent e) {
        
        if (e.getType() == javax.swing.event.TableModelEvent.UPDATE) {
            int row = e.getFirstRow();
           
            if (table.getValueAt(row, 0) == (Object)true) {
                if (!selectedRows.contains(row)) {
                    selectedRows.add(row);
                }
            } else {
                if (selectedRows.contains(row)) {
                    selectedRows.remove((Integer)row);
                }   
            }
        } else if (e.getType() == javax.swing.event.TableModelEvent.INSERT || e.getType() == javax.swing.event.TableModelEvent.DELETE) {
            project.setSaved(false);
        }
    } 
   
    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JButton addButton;
    private javax.swing.JButton analyseButton;
    private javax.swing.JPanel buttonsPanel;
    private javax.swing.JButton deselectAllButton;
    private javax.swing.JScrollPane faceTableScrollPanel;
    private javax.swing.JPanel filterPanel;
    private javax.swing.JButton inflateButton;
    private javax.swing.JPanel infoPanel;
    private javax.swing.JPanel mainPanel;
    private javax.swing.JScrollPane mainScrollPanel;
    private javax.swing.JButton manyToManyButton;
    private javax.swing.JButton newProjectButton;
    private javax.swing.JButton oneOnOneButton;
    private javax.swing.JButton openProjectButton;
    private javax.swing.JButton removeButton;
    private javax.swing.JButton saveProjectButton;
    private javax.swing.JButton selectAllButton;
    private javax.swing.JTable table;
    // End of variables declaration//GEN-END:variables
    @Override
    public void componentOpened() {
        // TODO add custom code on component opening
    }

    @Override
    public void componentClosed() {
        // TODO add custom code on component closing
    }

    void writeProperties(java.util.Properties p) {
        // better to version settings since initial version as advocated at
        // http://wiki.apidesign.org/wiki/PropertyFiles
        p.setProperty("version", "1.0");
        // TODO store your settings
    }

    void readProperties(java.util.Properties p) {
        String version = p.getProperty("version");
        // TODO read your settings according to their version
    }

    /**
     * Loads model selected in file chooser by user
     * @param loadFromFile true if models are loaded from file, else user chooses
     * from FileChooserBuilder
     */
    public void loadModel(boolean loadFromFile) {
        File[] files;
        
        // Selects files with faces (by dialog or from JSON file)
        if (loadFromFile) {
            files = new FileChooserBuilder(ProjectTopComp.class)
                .setTitle("Open human face(s)")
                .setDefaultWorkingDirectory(new File(System.getProperty("user.home")))
                //.setApproveText("Add")
                .setFileFilter(new FileNameExtensionFilter("obj files (*.obj)", "obj"))
                .setAcceptAllFileFilterUsed(true)
                .showMultiOpenDialog();           
        } else {
            files = new File[project.getCfg().getPaths().size()];
            files = project.getCfg().openFiles().toArray(files);
            project.getCfg().clearPaths();
        }
        
        if (files == null) {
            System.out.print("No file chosen.");
        } else {
            for (File file : files) {
                
                Path path = Paths.get(file.getAbsolutePath());
                String name = path.toString().substring(path.toString().lastIndexOf(File.separatorChar) + 1, path.toString().lastIndexOf('.'));
                if (project.addNewPath(path)) {  
                    model.addRowWithName(name, false);
                } else {
                    JOptionPane.showMessageDialog(this, String.format("Model with name %s is already loaded", name));
                }
            }
        }
    }
    
    /**
     * Loads tabs from project file
     */
    private void loadTabs() {
        
        project.getCfg().getSingleTabFaces().forEach(name -> {
            HumanFace face1 = project.loadFace(name);
            createSingleFaceTab(face1, name, true);
        });
        
        project.getCfg().getFaceToFaceTabFaces().keySet().forEach(name -> {
            HumanFace face1 = project.loadFace(name);
            project.getCfg().getFaceToFaceTabFaces().get(name).forEach(otherName -> {
                HumanFace face2 = project.loadFace(otherName);
                createFaceToFaceTab(face1, face2, name + ":" + otherName, true);
            });
        });
        
        this.toFront();
        this.openAtTabPosition(0);
        this.requestActive();
        
    }

   /**
    * Creates and opens tab with one face
    * @param face which will be analyzed
    * @param name name of the tab (name of the model)
    */
    private void createSingleFaceTab(HumanFace face, String name, boolean loadFromFile) {
        ActionListener tabCloseListener = new AbstractAction() {
            @Override
            public void actionPerformed(ActionEvent e) {
                closeTab(e);
            }
        };
        FaceTab newTab = new FaceTab(face, null, name, tabCloseListener);
        
        if (!tabs.contains(newTab)) {
            tabs.add(newTab);
            if (!loadFromFile) {
                project.addNewSingleFaceTabFace(name);
            }
            newTab.open();
            newTab.requestActive();
            this.project.setSaved(false);    
        }
    }

    /**
     * Creates and opens tab with two faces (1:1 analysis)
     * @param face1 which will be analyzed
     * @param face2 which will be analyzed
     * @param name name of the tab
     */
    private void createFaceToFaceTab(HumanFace face1, HumanFace face2, String nameOfTab, boolean loadFromFile) {
        ActionListener tabCloseListener = new AbstractAction() {
            @Override
            public void actionPerformed(ActionEvent e) {
                closeTab(e);
            }
        };
        
        FaceTab newTab = new FaceTab(face1, face2, nameOfTab, tabCloseListener);
        
        if (!tabs.contains(newTab)) {
            tabs.add(newTab);

            if (!loadFromFile) {
                project.addNewFaceToFaceTabFace(face1.getShortName(), face2.getShortName());
            }
            newTab.open();
            newTab.requestActive();
            this.project.setSaved(false);
        }
    }
    
    /**
     * Creates and opens tab with multiple faces for N:N analysis
     * @param faces faces to be analyzed
     * @param name name of the tab
     */
    private void createManyToManyTab(String name) {
        ActionListener tabCloseListener = new AbstractAction() {
            @Override
            public void actionPerformed(ActionEvent e) {
                closeTab(e);
            }
        };
        FaceTab newTab = new FaceTab(null, null, name, tabCloseListener);
        if (!tabs.contains(newTab)) {
            tabs.add(newTab);
        }
        newTab.open();
        newTab.requestActive();
    }

    /**
     * Opens info panel with face state information
     * @param faceName String name of face
     */
    private void checkFaceState(String faceName) {
        HumanFace face = project.getFaceByName(faceName);
        //FaceStatePanel fsp = new FaceStatePanel(infoPanel, face);
        
    }
    
    /**
     * Sorts faces by alphabet
     */
    private void alphabeticalFilter() {
        this.deselectAllRows();
        /*
        TableRowSorter<TableModel> sorter = new TableRowSorter<>(model);
        jTable1.setRowSorter(sorter);
        
        List<RowSorter.SortKey> sortKeys = new ArrayList<>();
        sortKeys.add(new RowSorter.SortKey(1, SortOrder.ASCENDING));
        sorter.setSortKeys(sortKeys);*/
        
        List<String> names = new ArrayList();
        while (model.getRowCount() > 0) {
            names.add(model.getValueAt(0, 1).toString());
            model.removeRow(0);
        }
        Collections.sort(names);
        names.forEach(name -> {
            HumanFace face = project.getFaceByName(name);
            model.addRowWithName(name, face.hasKdTree());
        });
        
    }
    
    /**
     * Removes faces from project (and table of faces) based on filter configuration
     */
    private void applyFilter() {
        deselectAllRows();
                
        for (int i = 0; i < model.getRowCount(); i++) { 
            HumanFace face = project.getFaceByName(model.getValueAt(i, 1).toString());
            
            if ((fp.isKdTreeFilter() && !face.hasKdTree()) || (fp.isFeaturePointsFilter() && !face.hasFeaturePoints())) {
                model.setValueAt(true, i, 0);
            }
        } 
        removeSelectedFaces();
        
        if (fp.isAlphaBeticalFilter()) {
            alphabeticalFilter();
        }
    }
    
    /**
     * Asks user whether he wants to create new empty project, or open existing
     * project
     */
    private void openExistingOrNewProject() {
        ImageIcon newProjectImage = new ImageIcon(ProjectTopComp.class.getClassLoader().getResource("/" + "new.png"));
        ImageIcon existingProjectImage = new ImageIcon(ProjectTopComp.class.getClassLoader().getResource("/" + "open.png"));
        Object[] options = {newProjectImage, existingProjectImage};
        
        
        int choice = JOptionPane.showOptionDialog(this, 
                "Would you like to create a new project or open an existing project?", 
                "Select project", 
                JOptionPane.DEFAULT_OPTION, JOptionPane.QUESTION_MESSAGE, 
                null, 
                options, 
                options[0]);
        
        if (choice == 1) {
            openProject();
        }
    }
    
    /**
     * Resets all project attributes
     */
    private void resetProject() {
        
        while (!tabs.isEmpty()) {
            tabs.get(0).close();
        }
        project.removeAll();
        model.setRowCount(0);
        selectedRows.clear();        
    }
    
    /**
     * Checks whether current project is saved, if not then asks user if he wants
     * to save it
     * @return 
     */
    private boolean loadNewProject() {
        if (!project.isSaved()) {
            int showConfirmDialog = JOptionPane.showConfirmDialog(this, "Project is not saved. Would you like to save project?", "Save project", JOptionPane.YES_NO_CANCEL_OPTION);
            
            switch (showConfirmDialog) {
                
                case JOptionPane.YES_OPTION:
                    saveProject();
                    break;
                case JOptionPane.CANCEL_OPTION:
                case JOptionPane.CLOSED_OPTION:
                    return false;
                default:
                    break;
            }
        }
        return true;
    }
    
    /**
     * Saves current project
     */
    private void saveProject() {
        
        JFileChooser chooser = new JFileChooser();
        //chooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
        chooser.setFileFilter(new FileNameExtensionFilter("json files (*.json)", "json"));
        chooser.setAcceptAllFileFilterUsed(true);
        chooser.showSaveDialog(null);
        
        File file = chooser.getSelectedFile();
        
        if (file != null) {
            String filePath = file.getAbsolutePath();
            if (!filePath.endsWith(".json")) {
                file = new File(filePath.concat(".json"));
            }

            try {
                mapper.writeValue(file, project.getCfg());
                project.setSaved(true);
            } catch (IOException ex) {
                Exceptions.printStackTrace(ex);
            }
        }
        
   
    }
    /**
     * Opens project, which user selects in File Chooser
     */
    private void openProject() {
        
        File f;
        f = new FileChooserBuilder(ProjectTopComp.class)
                .setTitle("Choose existing project")
                .setDefaultWorkingDirectory(new File(System.getProperty("user.home")))
                .setFileFilter(new FileNameExtensionFilter("json files (*.json)", "json"))
                .setAcceptAllFileFilterUsed(true)
                .showOpenDialog();

        if (f != null) {
            try {
                resetProject();
                project.setCfg(mapper.readValue(f, ProjectConfiguration.class));
                loadModel(false);    
                loadTabs();
                project.setSaved(true);
            } catch (IOException ex) {
                Exceptions.printStackTrace(ex);
            }

        }  
    }
    
    /**
     * Creates new project
     */
    private void newProject() {    
        resetProject();
        project.setCfg(new ProjectConfiguration());
        project.setSaved(true);
    }

    /**
     * Closes tab
     * @param e ActionEvent
     */
    private void closeTab(ActionEvent e) {
        FaceTab tab = (FaceTab)e.getSource();
        
        for (FaceTab t : tabs) {
            if (t.equals(tab)) {
                          
                if (e.getActionCommand().equals("FaceTab")) {
                    this.project.removeFaceTab(t.getNameOfFace1());
                } else if (e.getActionCommand().equals("FaceToFaceTab")){
                    this.project.removeFaceToFaceTab(t.getNameOfFace1(), t.getNameOfFace2());
                }
                tabs.remove(t);
                break;
            }
        }
        
        this.requestActive();
    }
    
    @Override
    public boolean canClose() {
        if (!project.isSaved()) {

            int save = JOptionPane.showConfirmDialog(null,
                    "Project is not saved. Would you like to save project?", "Save project", JOptionPane.YES_NO_OPTION, JOptionPane.QUESTION_MESSAGE);

            if (save == JOptionPane.YES_OPTION) {
                saveProject();
            }
        }
        return super.canClose(); //To change body of generated methods, choose Tools | Templates.
    }

}
