package cz.fidentis.analyst.canvas.toolbar;

import cz.fidentis.analyst.canvas.Canvas;
import cz.fidentis.analyst.scene.DrawableFace;
import cz.fidentis.analyst.scene.DrawableFeaturePoints;
import java.awt.Color;
import java.awt.FlowLayout;
import java.awt.event.ActionEvent;
import javax.swing.AbstractAction;
import javax.swing.ImageIcon;
import javax.swing.JPanel;
import javax.swing.JSlider;
import javax.swing.JToggleButton;
import javax.swing.event.ChangeEvent;
import javax.swing.plaf.metal.MetalToggleButtonUI;
import org.openide.util.NbBundle;

/**
 * Canvas toolbox for scenes with two faces. 
 * It enables to show-hide faces in the scene, show-hide feature points,
 * and to change the transparency of face couple.
 * 
 * @author Richard Pajersky
 * @author Radek Oslejsek
 */
public class SceneToolboxFaceToFace extends JPanel {
    
    private static final String PRIMARY_FACE_ICON = "primaryFace28x28.png";
    private static final String SECONDARY_FACE_ICON = "secondaryFace28x28.png";
    private static final String FEATURE_POINTS_ICON = "fps28x28.png";
    private static final String DISTANCE_BUTTON_ICON = "distance28x28.png";
    
    private static final int TRANSPARENCY_RANGE = 50;
    
    private JToggleButton primLandButton;
    private JToggleButton secLandButton;
    private JToggleButton primFaceButton;
    private JToggleButton secFaceButton;
    private JSlider       slider;
    private JToggleButton secDistButton;
    
    private final Canvas canvas;
    
    /**
     * Constructor.
     * @param canvas Rendering canvas
     */
    public SceneToolboxFaceToFace(Canvas canvas) {
        this.canvas = canvas;
        initComponents();
        
        // Change inital state:
        primLandButton.setSelected(false);
        DrawableFeaturePoints fp0 = canvas.getScene().getDrawableFeaturePoints(canvas.getScene().getPrimaryFaceIndex());
        if (fp0 != null) {
            fp0.show(false);
        }
        
        secLandButton.setSelected(false);
        DrawableFeaturePoints fp1 = canvas.getScene().getDrawableFeaturePoints(canvas.getScene().getSecondaryFaceIndex());
        if (fp1 != null) {
            fp1.show(false);
        }
        
        secDistButton.setSelected(true);
        canvas.getScene().getDrawableFace(canvas.getScene().getSecondaryFaceIndex()).setRenderHeatmap(true);
        
        slider.setValue(30);
        canvas.getScene().getDrawableFace(canvas.getScene().getPrimaryFaceIndex()).setTransparency(30/(float)TRANSPARENCY_RANGE);
        canvas.getScene().getDrawableFace(canvas.getScene().getSecondaryFaceIndex()).setTransparency(1);
    }

    private void initComponents() {
        JPanel panel = new JPanel();
        panel.setLayout(new FlowLayout());
        add(panel);
        
        primLandButton = new JToggleButton();
        primLandButton.setBorder(Canvas.BUTTONS_BORDER);
        primLandButton.setBackground(DrawableFace.SKIN_COLOR_PRIMARY);
        primLandButton.setUI(new MetalToggleButtonUI() {
            @Override
            protected Color getSelectColor() {
                return DrawableFace.SKIN_COLOR_PRIMARY.darker();
            }
        });
        primLandButton.setIcon(new ImageIcon(getClass().getResource("/" + FEATURE_POINTS_ICON)));
        primLandButton.setFocusable(false);
        primLandButton.setSelected(true);
        primLandButton.setToolTipText(NbBundle.getMessage(SceneToolboxFaceToFace.class, "SceneToolboxFaceToFace.landButton.tooltip"));
        panel.add(primLandButton);
        
        primLandButton.addActionListener(new AbstractAction() {
            @Override
            public void actionPerformed(ActionEvent e) {
                boolean onOff = ((JToggleButton) e.getSource()).isSelected();
                canvas.getScene().getDrawableFeaturePoints(canvas.getScene().getPrimaryFaceIndex()).show(onOff);
                canvas.renderScene();
            }
        });

        /////////////////////////
        primFaceButton = new JToggleButton();
        primFaceButton.setBorder(Canvas.BUTTONS_BORDER);
        primFaceButton.setBackground(DrawableFace.SKIN_COLOR_PRIMARY);
        primFaceButton.setUI(new MetalToggleButtonUI() {
            @Override
            protected Color getSelectColor() {
                return DrawableFace.SKIN_COLOR_PRIMARY.darker();
            }
        });
        primFaceButton.setIcon(new ImageIcon(getClass().getResource("/" + PRIMARY_FACE_ICON)));
        primFaceButton.setFocusable(false);
        primFaceButton.setSelected(true);
        primFaceButton.setToolTipText(NbBundle.getMessage(SceneToolboxFaceToFace.class, "SceneToolboxFaceToFace.faceButton.tooltip"));
        panel.add(primFaceButton);
        
        primFaceButton.addActionListener(new AbstractAction() {
            @Override
            public void actionPerformed(ActionEvent e) {
                boolean onOff = ((JToggleButton) e.getSource()).isSelected();
                canvas.getScene().showDrawableFace(canvas.getScene().getPrimaryFaceIndex(), onOff);
                canvas.renderScene();
            }
        });
        
        /////////////////////////
        slider = new JSlider();
        slider.setMajorTickSpacing(TRANSPARENCY_RANGE/2);
        slider.setMaximum(TRANSPARENCY_RANGE*2);
        slider.setMinorTickSpacing(TRANSPARENCY_RANGE/2);
        slider.setPaintTicks(false);
        slider.setToolTipText(org.openide.util.NbBundle.getMessage(SceneToolboxFaceToFace.class, "SceneToolboxFaceToFace.slider.tooltip"));
        slider.setValue(TRANSPARENCY_RANGE);
        slider.setCursor(new java.awt.Cursor(java.awt.Cursor.HAND_CURSOR));
        panel.add(slider);
        
        slider.addChangeListener((ChangeEvent e) -> {
            int val = ((JSlider) e.getSource()).getValue();
            if (val <= TRANSPARENCY_RANGE) {
                canvas.getScene().getDrawableFace(canvas.getScene().getPrimaryFaceIndex())
                        .setTransparency(val/(float)TRANSPARENCY_RANGE);
                canvas.getScene().getDrawableFace(canvas.getScene().getSecondaryFaceIndex()).setTransparency(1);
            } else if (val > TRANSPARENCY_RANGE) {
                canvas.getScene().getDrawableFace(canvas.getScene().getPrimaryFaceIndex()).setTransparency(1);
                canvas.getScene().getDrawableFace(canvas.getScene().getSecondaryFaceIndex())
                        .setTransparency((2 * TRANSPARENCY_RANGE - val) / (float)TRANSPARENCY_RANGE);
            }
            canvas.renderScene();
        });
        
        /////////////////////////
        secFaceButton = new JToggleButton();
        secFaceButton.setBorder(Canvas.BUTTONS_BORDER);
        secFaceButton.setBackground(DrawableFace.SKIN_COLOR_SECONDARY);
        secFaceButton.setUI(new MetalToggleButtonUI() {
            @Override
            protected Color getSelectColor() {
                return DrawableFace.SKIN_COLOR_SECONDARY.darker();
            }
        });
        secFaceButton.setIcon(new ImageIcon(getClass().getResource("/" + SECONDARY_FACE_ICON)));
        secFaceButton.setFocusable(false);
        secFaceButton.setSelected(true);
        secFaceButton.setToolTipText(NbBundle.getMessage(SceneToolboxFaceToFace.class, "SceneToolboxFaceToFace.faceButton.tooltip"));
        panel.add(secFaceButton);
        
        secFaceButton.addActionListener(new AbstractAction() {
            @Override
            public void actionPerformed(ActionEvent e) {
                boolean onOff = ((JToggleButton) e.getSource()).isSelected();
                canvas.getScene().showDrawableFace(canvas.getScene().getSecondaryFaceIndex(), onOff);
                canvas.renderScene();
            }
        });
        

        /////////////////////////
        secLandButton = new JToggleButton();
        secLandButton.setBorder(Canvas.BUTTONS_BORDER);
        secLandButton.setBackground(DrawableFace.SKIN_COLOR_SECONDARY);
        secLandButton.setUI(new MetalToggleButtonUI() {
            @Override
            protected Color getSelectColor() {
                return DrawableFace.SKIN_COLOR_SECONDARY.darker();
            }
        });
        secLandButton.setIcon(new ImageIcon(getClass().getResource("/" + FEATURE_POINTS_ICON)));
        secLandButton.setFocusable(false);
        secLandButton.setSelected(true);
        secLandButton.setToolTipText(NbBundle.getMessage(SceneToolboxFaceToFace.class, "SceneToolboxFaceToFace.landButton.tooltip"));
        panel.add(secLandButton);
        
        secLandButton.addActionListener(new AbstractAction() {
            @Override
            public void actionPerformed(ActionEvent e) {
                boolean onOff = ((JToggleButton) e.getSource()).isSelected();
                canvas.getScene().getDrawableFeaturePoints(canvas.getScene().getSecondaryFaceIndex()).show(onOff);
                canvas.renderScene();
            }
        });

        /////////////////////
        secDistButton = new JToggleButton();
        secDistButton.setBorder(Canvas.BUTTONS_BORDER);
        secDistButton.setBackground(DrawableFace.SKIN_COLOR_SECONDARY);
        secDistButton.setUI(new MetalToggleButtonUI() {
            @Override
            protected Color getSelectColor() {
                return DrawableFace.SKIN_COLOR_SECONDARY.darker();
            }
        });
        secDistButton.setIcon(new ImageIcon(getClass().getResource("/" + DISTANCE_BUTTON_ICON)));
        secDistButton.setFocusable(false);
        secDistButton.setSelected(true);
        secDistButton.setToolTipText(NbBundle.getMessage(SceneToolboxFaceToFace.class, "SceneToolboxFaceToFace.distButton.tooltip"));
        panel.add(secDistButton);
        
        secDistButton.addActionListener(new AbstractAction() { 
            @Override
            public void actionPerformed(ActionEvent e) {
                if (((JToggleButton) e.getSource()).isSelected()) {
                    canvas.getScene().getDrawableFace(canvas.getScene().getSecondaryFaceIndex()).setRenderHeatmap(true);
                } else {
                    canvas.getScene().getDrawableFace(canvas.getScene().getSecondaryFaceIndex()).setRenderHeatmap(false);
                }
                // Switch:
                //canvas.getScene().getDrawableFace(secIndex).setRenderHeatmap(
                //        !canvas.getScene().getDrawableFace(secIndex).isHeatmapRendered()
                //);
                canvas.renderScene();
            }
        });
    }
    
}
