package cz.fidentis.analyst.batch;

import com.jogamp.opengl.GL2;
import cz.fidentis.analyst.Logger;
import cz.fidentis.analyst.canvas.Canvas;
import cz.fidentis.analyst.core.ControlPanelAction;
import cz.fidentis.analyst.face.HumanFace;
import cz.fidentis.analyst.face.events.HumanFaceEvent;
import cz.fidentis.analyst.face.events.HumanFaceListener;
import cz.fidentis.analyst.core.ProgressDialog;
import cz.fidentis.analyst.project.ProjectTopComp;
import cz.fidentis.analyst.scene.DrawableFace;
import java.awt.event.ActionEvent;
import java.beans.PropertyChangeEvent;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import javax.swing.JOptionPane;
import javax.swing.JTabbedPane;
import javax.swing.SwingWorker;
import javax.swing.filechooser.FileNameExtensionFilter;
import org.openide.filesystems.FileChooserBuilder;

/**
 * Action listener for batch registration phase.
 * 
 * @author Radek Oslejsek
 */
public class BatchAction extends ControlPanelAction implements HumanFaceListener {
    
    private final BatchPanel controlPanel;
    private int faceSceneSlot = -1;
    
    private double[][] distances;
    
    /**
     * Constructor.
     *
     * @param canvas OpenGL canvas
     * @param topControlPanel Top component for placing control panels
     */
    public BatchAction(Canvas canvas, JTabbedPane topControlPanel) {
        super(canvas, topControlPanel);
        this.controlPanel = new BatchPanel(this);
        
        // Place control panel to the topControlPanel
        topControlPanel.addTab(controlPanel.getName(), controlPanel.getIcon(), controlPanel);
        topControlPanel.addChangeListener(e -> {
            // If the registration panel is focused...
            if (((JTabbedPane) e.getSource()).getSelectedComponent() instanceof BatchPanel) {
                // TODO
            }
        });
        topControlPanel.setSelectedComponent(controlPanel); // Focus registration panel
        
        // Be informed about changes in faces perfomed by other GUI elements
        //getPrimaryDrawableFace().getHumanFace().registerListener(this);
        //getSecondaryDrawableFace().getHumanFace().registerListener(this);
    }

    @Override
    public void actionPerformed(ActionEvent ae) {
        String action = ae.getActionCommand();
         switch (action) {
            case BatchPanel.ACTION_COMMAND_COMPUTE_ICP:
                computeRegistrationAndTemplate();
                break;
            case BatchPanel.ACTION_COMMAND_COMPUTE_SIMILARITY:
                computeSimilarity();
                break;
            case BatchPanel.ACTION_COMMAND_SHOW_SELECTED_FACE:
                showSelectedIcpFace();
                break;
            case BatchPanel.ACTION_COMMAND_EXPORT_SIMILARITY:
                exportDistanceResults();
                break;
            default:
         }
        renderScene();
    }

    @Override
    public void acceptEvent(HumanFaceEvent event) {
        // NOTHING TO DO
    }

    private void computeRegistrationAndTemplate() {
        if (controlPanel.getSelectedFaceIndex() == -1) {
            JOptionPane.showMessageDialog(
                    controlPanel,
                    "No faces in the data set",
                    null,
                    JOptionPane.WARNING_MESSAGE);
            return;
        }
        
        ProgressDialog progressBar = new ProgressDialog(controlPanel, "ICP registration and average face computation");
        
        IcpTask task = new IcpTask(
                controlPanel.getSelectedFaceIndex(),
                progressBar, 
                controlPanel, 
                getCanvas());
        
        // The following code will be executed when the task is done:
        task.addPropertyChangeListener((PropertyChangeEvent evt) -> { 
            if ("state".equals(evt.getPropertyName()) && (SwingWorker.StateValue.DONE.equals(evt.getNewValue()))) {
                HumanFace avgFace = task.getAverageFace();
                getScene().clearScene();
                faceSceneSlot = -1;
                getCanvas().getCamera().initLocation();
                controlPanel.addAndSelectAvgFace(avgFace); // show selected face (either the average ot other)
            }
        });

        // Prepare the scene - show the selected face towards which other faces are transformed:
        HumanFace selectedFace = controlPanel.getSelectedFace();
        if (controlPanel.showIcpPreview() && selectedFace != null) {
            getScene().clearScene(); // scene is recovered automatically - see the event handler above
            faceSceneSlot = getScene().getFreeSlot();
            
            // Add inital face to the scene:
            getScene().setDrawableFace(faceSceneSlot, selectedFace);
            getScene().getDrawableFace(faceSceneSlot).setTransparency(0.7f);
            getScene().getDrawableFace(faceSceneSlot).setRenderMode(GL2.GL_POINT);
            getScene().getDrawableFace(faceSceneSlot).setColor(DrawableFace.SKIN_COLOR_PRIMARY);
            
            // locate the camera to the best angle:
            getCanvas().getCamera().initLocation(); 
            getCanvas().getCamera().rotate(10, -80);
            getCanvas().getCamera().move(40, 20);
        }
        
        progressBar.runTask(task);
    } 
    
    private void computeSimilarity() {
        ProgressDialog progressBar = new ProgressDialog(controlPanel, "Similarity computation");
        final SimilarityTask task;
        
        switch (controlPanel.getSimilarityStrategy()) {
            case BatchPanel.SIMILARITY_COMPLETE_HD:
                task = new CompleteHausdorffDistTask(progressBar, controlPanel);
                break;
            case BatchPanel.SIMILARITY_APPROX_HD:
                task = new ApproxHausdorffDistTask(progressBar, controlPanel, controlPanel.getSelectedFaceIndex());
                break;
            default:
                return;
        }
        
        task.addPropertyChangeListener((PropertyChangeEvent evt) -> { // when done ...
            if ("state".equals(evt.getPropertyName()) && (SwingWorker.StateValue.DONE.equals(evt.getNewValue()))) {
                double[][] result = task.getDistSimilarities();
                if (result != null) {
                    this.distances = result;
                }
                controlPanel.enableSimilatiryExport(this.distances != null);
           }
        });
        
        progressBar.runTask(task);
    } 

    private void showSelectedIcpFace() {
        HumanFace face = controlPanel.getSelectedFace();
        if (face != null) {
            if (faceSceneSlot == -1) {
                faceSceneSlot = getScene().getFreeSlot();
            }
            getScene().setDrawableFace(faceSceneSlot, face); 
            getScene().setFaceAsPrimary(faceSceneSlot);
            getScene().showDrawableFace(faceSceneSlot, true);
        }
    }

    private void exportDistanceResults() {
        if (this.distances == null) {
            return;
        }
        
        File file = new FileChooserBuilder(ProjectTopComp.class)
                .setTitle("Specify a file to save")
                .setDefaultWorkingDirectory(new File(System.getProperty("user.home")))
                .setFileFilter(new FileNameExtensionFilter("CSV files (*.csv)", "csv"))
                .showSaveDialog();
        
        if (file != null) {
            try {
                BufferedWriter w = new BufferedWriter(new FileWriter(file));
                w.write("PRI FACE;SEC FACE;Dist from PRI to SEC;Dist from SEC to PRI;lower HD;higher HD");
                w.newLine();
                
                for (int i = 0; i < distances.length; i++) {
                    for (int j = i; j < distances.length; j++) {
                        if (distances[i][j] != distances[j][i]) {
                            Logger.print("SIMILARITY ERROR for face " + i + " and " + j);
                        }
                        w.write(getShortName(controlPanel.getFacePaths().get(i).toFile()) + ";");
                        w.write(getShortName(controlPanel.getFacePaths().get(j).toFile()) + ";");
                        w.write(String.format("%.8f", distances[i][j]) + ";");
                        w.write(String.format("%.8f", distances[j][i]) + ";");
                        if (distances[i][j] > distances[j][i]) {
                            w.write(String.format("%.8f", distances[i][j]) + ";");
                            w.write(String.format("%.8f", distances[j][i]) + "");
                        } else {
                            w.write(String.format("%.8f", distances[j][i]) + ";");
                            w.write(String.format("%.8f", distances[i][j]) + "");
                        }
                        w.newLine();
                    }
                }

                w.close();
            } catch (IOException ex) {
                Logger.print(ex.toString());
            }
        }
    }
    
    private static String getShortName(File file) throws IOException {
        String name = file.toString();
        name = name.substring(name.lastIndexOf(File.separatorChar) + 1, name.length());
        return name;
    }

    
}
