package cz.fidentis.analyst.mesh.core;

import cz.fidentis.analyst.mesh.visitors.BoundingBox;
import cz.fidentis.analyst.mesh.visitors.TriangleListVisitor;
import cz.fidentis.analyst.mesh.visitors.Visitor;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.vecmath.Vector3d;

/**
 * MashFacet
 *
 * @author Matej Lukes
 */
public class MeshFacetImpl implements MeshFacet {
    
    private List<MeshPoint> vertices = new ArrayList<>();
    private CornerTable cornerTable;

    /**
     * Constructor of MeshFacet
     */
    public MeshFacetImpl() {
        cornerTable = new CornerTable();
    }

    /**
     * Copy constructor of MeshFacet
     *
     * @param facet copied MeshFacet
     */
    public MeshFacetImpl(MeshFacet facet) {
        vertices.addAll(facet.getVertices()); // encapsulation preserved - vertices MeshPoints are immutable)
        cornerTable = new CornerTable(facet.getCornerTable());
    }
    
    @Override
    public void accept(Visitor visitor) {
        visitor.visitMeshFacet(this);
    }

    @Override
    public MeshPoint getVertex(int index) {
        return vertices.get(index);
    }

    @Override
    public void addVertex(MeshPoint point) {
        vertices.add(point);
    }

    @Override
    public int getNumberOfVertices() {
        return vertices.size();
    }

    @Override
    public List<MeshPoint> getVertices() {
        return Collections.unmodifiableList(vertices);
    }

    @Override
    public CornerTable getCornerTable() {
        return cornerTable;
    }
    
    @Override
    public boolean hasVertexNormals() {
        return !this.vertices.isEmpty() && this.vertices.get(0).getNormal() != null;
    }
    
    /**
     * REPLACE WITH BETTER IMPLEMENTATION
     * @throws RuntimeException if there are duplicate meth points in the mesh facet
     */
    @Override
    public void calculateVertexNormals() {
        Map<Vector3d, Vector3d> normalMap = new HashMap<>(); // key = mesh point, value = normal
        
        // init normals:
        for (MeshPoint point: vertices) { 
            if (normalMap.put(point.getPosition(), new Vector3d(0, 0, 0)) != null) {
                throw new RuntimeException("Duplicate mesh point in the MeshFacet: " + point);
            }
        }
        
        // calculate normals from corresponding triangles
        TriangleListVisitor visitor = new TriangleListVisitor();
        this.accept(visitor);
        for (MeshTriangle t : visitor.getTriangles()) { 
            Vector3d triangleNormal = 
                    (t.vertex3.subtractPosition(t.vertex1)).crossProduct(t.vertex2.subtractPosition(t.vertex1)).getPosition();
            normalMap.get(t.vertex1).add(triangleNormal);
            normalMap.get(t.vertex2).add(triangleNormal);
            normalMap.get(t.vertex3).add(triangleNormal);
        }
        
        // normalize normals:
        for (Vector3d normal: normalMap.values()) { 
            normal.normalize();
        }
    }

}

