package cz.fidentis.analyst.feature.provider;

import cz.fidentis.analyst.feature.FeaturePointType;
import cz.fidentis.analyst.feature.services.FeaturePointTypesService;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

/**
 * Feature point type provider class
 *
 * @author Jakub Kolman
 */
public final class FeaturePointTypeProvider {

    private final Map<Integer, FeaturePointType> featurePointTypesById;
    private final Map<String, FeaturePointType> featurePointTypesByCode;

    /**
     * provide thread safe singleton
     *
     * @author Jakub Kolman
     */
    private static class InstanceHolder {
        public static final FeaturePointTypeProvider INSTANCE = new FeaturePointTypeProvider();
    }

    /**
     *
     */
    private FeaturePointTypeProvider() {
        FeaturePointTypesService service = new FeaturePointTypesService();
        featurePointTypesById = service.readResources().orElse(new HashMap<>());
        featurePointTypesByCode = service.getFeaturepointTypesByCode(featurePointTypesById);
    }

    /**
     * Access feature point types
     *
     * @return
     */
    public static FeaturePointTypeProvider getInstance() {
        return InstanceHolder.INSTANCE;
    }

    /**
     * get feature points types by id
     *
     * @return
     */
    public Map<Integer, FeaturePointType> getFeaturePointTypesById() {
        return Collections.unmodifiableMap(featurePointTypesById);
    }

    /**
     * sorted feature point types by code
     *
     * @return
     */
    public Map<String, FeaturePointType> getFeaturePointTypesByCode() {
        return Collections.unmodifiableMap(featurePointTypesByCode);
    }

    /**
     * get single feature point type by its code
     *
     * @param code
     * @return
     */
    public FeaturePointType getFeaturePointTypeByCode(String code) {
        return getFeaturePointTypesByCode().get(code);
    }

    /**
     * get single feature point type by its id
     *
     * @param id
     * @return
     */
    public FeaturePointType getFeaturePointTypeById(int id) {
        return getFeaturePointTypesById().get(id);
    }
}
