package cz.fidentis.analyst;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;
import java.io.PrintStream;
import java.text.SimpleDateFormat;
import java.time.Duration;
import java.util.logging.Level;

/**
 * A logging class that can be used from anywhere to display debugging or other
 * messages.
 * <p>
 * Messages are printed into the standard output unless the {@link #reader()} 
 * method is called. Then the messages are redirected to the returned {@code BufferedReader}.
 * </p>
 * <p>
 * The implementation supposes that the caller of the {@link #reader()} method 
 * (or other object} will read the messages from the {@code BufferedReader} forever. 
 * Otherwise, the program can freeze due to the full buffer.
 * </p>
 * <p>
 * FIDENTIS application automatically redirects logged messages to the output window.
 * </p>
 * <p>
 * <b>Usage:</b>
 * <ul>
 *    <li>To print a message, use {@code Logger.print("message")}. A timestamp is added automatically.</li>
 *    <li>To print message together with duration, call {@code Logger log = Logger.measureTime();}, 
 *        then call the measured operations and finally call {@code log.printDuration("message")}. 
 *        a timestamp and measured duration are added to the log message automatically.</li>
 * </ul>
 * </p>
 * 
 * @author Radek Oslejsek
 */
public final class Logger {
    
    private static PrintStream out = System.out;
    private static BufferedReader in;
    
    private long lastTime = System.currentTimeMillis();
    
    private Logger() {
    }
    
    /**
     * Redirects logs from standard output to the {@code BufferedReader}.
     * 
     * @return {@code BufferedReader} to which the logs are redirected.
     */
    public static BufferedReader reader() {
        if (in == null) {
            PipedOutputStream pOut = new PipedOutputStream();
            out = new PrintStream(pOut);
            try {
                in = new BufferedReader(new InputStreamReader(new PipedInputStream(pOut)));
            } catch (IOException ex) {
                java.util.logging.Logger.getLogger(Logger.class.getName()).log(Level.SEVERE, null, ex);
            }
        }
        return in;
    }
    
    /**
     * Prints a message. The log time added automatically.
     * 
     * @param msg Message to be logged
     */
    public static void print(String msg) {
        String outMsg = 
                new SimpleDateFormat("HH:mm:ss").format(System.currentTimeMillis())
                + " [duration unknown]: "
                + msg.trim();
        out.println(outMsg);
    }

    /**
     * Starts measuring of some operation. 
     * Call {@link #printDuration(java.lang.String)} on returned object
     * to print the measured duration.
     * 
     * @return An object used to print measured duration and message
     */
    public static Logger measureTime() {
        Logger ret = new Logger();
        ret.lastTime = System.currentTimeMillis();
        return ret;
    }
    
    /**
     * Prints the message about an operation and the duration of the operation. 
     * The duration is computed as the difference between current system time and 
     * the time of calling {@link #measureTime()}. The time difference is measured in 
     * milliseconds and printed as [+mm:ss.SSS] where mm=minutes, ss=seconds, 
     * and SSS=milliseconds.
     * 
     * @param msg Message to be printed
     */
    public void printDuration(String msg) {
        Duration duration = Duration.ofMillis(System.currentTimeMillis() - lastTime);
        String outMsg = 
                new SimpleDateFormat("HH:mm:ss").format(System.currentTimeMillis())
                + " [duration "
                + String.format(
                        "%02d:%02d.%03d", 
                        duration.toMinutes(), 
                        duration.toSecondsPart(),
                        duration.toMillisPart())
                + "]: "
                + msg.trim();
        
        out.println(outMsg);
    }
    
}
