package cz.fidentis.analyst.symmetry;

import cz.fidentis.analyst.canvas.Canvas;
import cz.fidentis.analyst.core.ControlPanelAction;
import cz.fidentis.analyst.visitors.mesh.CrossSection;
import org.openide.filesystems.FileChooserBuilder;

import javax.swing.JTabbedPane;
import javax.swing.JToggleButton;
import javax.swing.filechooser.FileNameExtensionFilter;
import javax.vecmath.Point3d;
import java.awt.event.ActionEvent;
import java.io.File;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.List;

/**
 * Action listener for the manipulation with the cutting plane.
 *
 * @author Dominik Racek
 */
public class ProfilesAction extends ControlPanelAction {

    /*
     * GUI elements
     */
    private final ProfilesPanel controlPanel;
    private final JTabbedPane topControlPanel;

    /*
     * Calculated profiles
     */
    private List<Point3d> primaryPoints;
    private List<Point3d> secondaryPoints;

    /**
     * Constructor.
     *
     * @param canvas          OpenGL canvas
     * @param topControlPanel Top component for placing control panels
     */
    public ProfilesAction(Canvas canvas, JTabbedPane topControlPanel) {
        super(canvas, topControlPanel);
        this.topControlPanel = topControlPanel;

        CrossSection cs1 = new CrossSection(getScene().getDrawableCuttingPlane(0).getFacets().get(0));
        getPrimaryDrawableFace().getModel().compute(cs1);
        this.primaryPoints = cs1.getPoints();

        if (getSecondaryDrawableFace() != null) {
            CrossSection cs2 = new CrossSection(getScene().getDrawableCuttingPlane(1).getFacets().get(0));
            getSecondaryDrawableFace().getModel().compute(cs2);
            this.secondaryPoints = cs2.getPoints();

            controlPanel = new ProfilesPanel(this, this.primaryPoints, this.secondaryPoints);
        } else {
            controlPanel = new ProfilesPanel(this, this.primaryPoints);
        }

        // Place control panel to the topControlPanel
        this.topControlPanel.addTab(controlPanel.getName(), controlPanel.getIcon(), controlPanel);
        this.topControlPanel.addChangeListener(e -> {
            // If the symmetry panel is focused...
            if (((JTabbedPane) e.getSource()).getSelectedComponent() instanceof ProfilesPanel) {
                getCanvas().getScene().setDefaultColors();
                getCanvas().getScene().showCuttingPlanes();
            } else {
                getCanvas().getScene().hideCuttingPlanes();
            }
        });
        getCanvas().getScene().hideCuttingPlanes();
    }

    private void exportProfile(List<Point3d> points, String title) {
        File file = new FileChooserBuilder(ProfilesAction.class)
                .setTitle(title)
                .setDefaultWorkingDirectory(new File(System.getProperty("user.home")))
                .setFilesOnly(true)
                .setFileFilter(new FileNameExtensionFilter("csv files (*.csv)", "csv"))
                .setAcceptAllFileFilterUsed(true)
                .showSaveDialog();

        if (file == null) {
            return;
        }

        // If chosen file exists, use the exact file path
        // If chosen file does not exist and does not have an extension, add it
        if (!file.exists()) {
            if (!file.getAbsolutePath().endsWith(".csv")) {
                file = new File(file.getAbsolutePath() + ".csv");
            }
        }

        try {
            file.createNewFile();

            PrintWriter writer = new PrintWriter(file.getAbsoluteFile(), "UTF-8");
            writer.println("N,X-Coordinate,Z-Coordinate");
            for (int i = 0; i < points.size(); ++i) {
                writer.println((i+1) + "," + points.get(i).x + "," + points.get(i).z);
            }

            writer.close();
        } catch (IOException ex) {
            System.out.println("ERROR writing to a file: " + ex);
        }
    }

    @Override
    public void actionPerformed(ActionEvent ae) {
        String action = ae.getActionCommand();

        switch (action) {
            case ProfilesPanel.ACTION_COMMAND_SHOW_HIDE_PLANE:
                if (((JToggleButton) ae.getSource()).isSelected()) {
                    getScene().showCuttingPlanes();
                } else {
                    getScene().hideCuttingPlanes();
                }
                break;
            case ProfilesPanel.ACTION_COMMAND_EXPORT:
                exportProfile(this.primaryPoints, "Export primary face profile to file");
                if (getSecondaryDrawableFace() != null) {
                    exportProfile(this.secondaryPoints, "Export secondary face profile to file");
                }
                break;
            default:
                // do nothing
        }
        renderScene();
    }

}
