package cz.fidentis.analyst.symmetry;

import cz.fidentis.analyst.canvas.Canvas;
import cz.fidentis.analyst.core.ControlPanelAction;
import cz.fidentis.analyst.core.DoubleSpinner;
import cz.fidentis.analyst.face.HumanFace;
import cz.fidentis.analyst.face.events.HumanFaceEvent;
import cz.fidentis.analyst.face.events.HumanFaceListener;
import cz.fidentis.analyst.face.events.HumanFaceTransformedEvent;
import cz.fidentis.analyst.scene.DrawableCuttingPlane;
import cz.fidentis.analyst.visitors.mesh.BoundingBox.BBox;
import cz.fidentis.analyst.visitors.mesh.CrossSection;
import org.openide.filesystems.FileChooserBuilder;

import javax.swing.JTabbedPane;
import javax.swing.filechooser.FileNameExtensionFilter;
import javax.vecmath.Point3d;
import java.awt.event.ActionEvent;
import java.io.File;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.List;
import java.util.stream.Collectors;
import javax.swing.JComboBox;
import javax.swing.JOptionPane;
import javax.vecmath.Vector3d;

/**
 * Action listener for the manipulation with the cutting plane.
 *
 * @author Dominik Racek
 * @author Radek Oslejsek
 */
public class ProfilesAction extends ControlPanelAction implements HumanFaceListener {
    
    public static final double FEATUE_POINTS_CLOSENESS = 1.5;

    /*
     * GUI elements
     */
    private final ProfilesPanel controlPanel;
    private final JTabbedPane topControlPanel;

    /*
     * Calculated profiles
     */
    private CrossSectionCurve primaryCurve;
    private CrossSectionCurve secondaryCurve;
    private CrossSectionCurve primaryMirrorCurve;
    private CrossSectionCurve secondaryMirrorCurve;

    private boolean cuttingPlaneFromSymmetry;
    
    private int priCuttingPlaneSlot = -1;
    private int secCuttingPlaneSlot = -1;    
    
    /**
     * Constructor.
     *
     * @param canvas          OpenGL canvas
     * @param topControlPanel Top component for placing control panels
     */
    public ProfilesAction(Canvas canvas, JTabbedPane topControlPanel) {
        super(canvas, topControlPanel);
        this.topControlPanel = topControlPanel;
        
        computeVerticalCuttingPlanes(); // compute default cutting planes

        //recomputePrimaryProfile(); 

        if (getSecondaryDrawableFace() != null) {
            //recomputeSecondaryProfile();
            controlPanel = new ProfilesPanel(this, this.primaryCurve, this.secondaryCurve);
            controlPanel.getProfileRenderingPanel().setSecondaryMirrorSegments(this.secondaryMirrorCurve);
        } else {
            controlPanel = new ProfilesPanel(this, this.primaryCurve);
        }
        controlPanel.getProfileRenderingPanel().setPrimaryMirrorSegments(this.primaryMirrorCurve);

        // Place control panel to the topControlPanel
        this.topControlPanel.addTab(controlPanel.getName(), controlPanel.getIcon(), controlPanel);
        this.topControlPanel.addChangeListener(e -> {
            // If the symmetry panel is focused...
            if (((JTabbedPane) e.getSource()).getSelectedComponent() instanceof ProfilesPanel) {
                getCanvas().getScene().setDefaultColors();
                showCuttingPlanes(true, controlPanel.isMirrorCutsChecked());
                recomputeProfiles();
            } else {
                showCuttingPlanes(false, false);
            }
        });
        
        showCuttingPlanes(false, false);
        
        // Be informed about changes in faces perfomed by other GUI elements
        getPrimaryDrawableFace().getHumanFace().registerListener(this);
        if (getSecondaryDrawableFace() != null) {
            getSecondaryDrawableFace().getHumanFace().registerListener(this);
        }
    }

    @Override
    public void actionPerformed(ActionEvent ae) {
        String action = ae.getActionCommand();
        
        switch (action) {
            case ProfilesPanel.ACTION_COMMAND_EXPORT:
                exportProfile(this.primaryCurve, "Export primary face profile to file");
                if (controlPanel.isMirrorCutsChecked()) {
                    exportProfile(this.primaryMirrorCurve, "Export primary face mirror profile to file");
                }

                if (getSecondaryDrawableFace() != null) {
                    exportProfile(this.secondaryCurve, "Export secondary face profile to file");
                    if (controlPanel.isMirrorCutsChecked()) {
                        exportProfile(this.secondaryMirrorCurve, "Export secondary face mirror profile to file");
                    }
                }
                break;
            case ProfilesPanel.ACTION_SHIFT_CUTTING_PLANE:
                double sliderVal = (Double) ((DoubleSpinner) ae.getSource()).getValue();
                double xExtent = maxExtentX();
                double shift = (sliderVal >= 0.5) 
                        ? ( (sliderVal - 0.5) / 0.5) * xExtent  // move right
                        : -((0.5 - sliderVal) / 0.5) * xExtent; // move left
                getDrawableCuttingPlane(priCuttingPlaneSlot).shift(shift);
                if (getSecondaryDrawableFace() != null) {
                    getDrawableCuttingPlane(secCuttingPlaneSlot).shift(shift);
                }
                recomputeProfiles();
                break;
            case ProfilesPanel.ACTION_CHANGE_CUTTING_PLANE:
                String option = (String)((JComboBox) ae.getSource()).getSelectedItem();
                
                if (option.equals(ProfilesPanel.OPTION_SYMMETRY_PLANE)) {
                    if (getScene().getDrawableSymmetryPlane(0) == null) {
                       JOptionPane.showMessageDialog(
                               controlPanel, 
                               "Compute a symmetry plane at the Symmetry tab first.", 
                               "No symmetry plane", 
                               JOptionPane.INFORMATION_MESSAGE
                       );
                       controlPanel.setDefaultPlaneSelection();
                       return;
                    }
                    controlPanel.resetSlider();
                    computeCuttingPlanesFromSymmetry();
                } else if (option.equals(ProfilesPanel.OPTION_VERTICAL_PLANE)) {
                    controlPanel.resetSlider();
                    computeVerticalCuttingPlanes();
                }
                
                showCuttingPlanes(true, controlPanel.isMirrorCutsChecked());
                recomputeProfiles();
                break;
            case ProfilesPanel.ACTION_MIRROR_CUTS:
                controlPanel.getProfileRenderingPanel().setMirrorCuts(controlPanel.isMirrorCutsChecked());
                showCuttingPlanes(true, controlPanel.isMirrorCutsChecked());
                break;
            default:
                // do nothing
        }
        renderScene();
    }
    
    @Override
    public void acceptEvent(HumanFaceEvent event) {
        // If some human face is transformed, then cutting planes have to updated.
        if (event instanceof HumanFaceTransformedEvent) {
            //controlPanel.resetSliderSilently();
            controlPanel.resetSlider();
            if (cuttingPlaneFromSymmetry) {
                computeCuttingPlanesFromSymmetry(); // recompute cutting planes
                //getCanvas().getScene().showCuttingPlanes(false, controlPanel.isMirrorCutsChecked());
            } else {
                computeVerticalCuttingPlanes();
            }
            getDrawableCuttingPlane(priCuttingPlaneSlot).show(false); // will be shown when the panel is focused
            if (getDrawableCuttingPlane(secCuttingPlaneSlot) != null) {
                getDrawableCuttingPlane(secCuttingPlaneSlot).show(false);
            }
        }
    }

    private void exportProfile(CrossSectionCurve curve, String title) {
        File file = new FileChooserBuilder(ProfilesAction.class)
                .setTitle(title)
                .setDefaultWorkingDirectory(new File(System.getProperty("user.home")))
                .setFilesOnly(true)
                .setFileFilter(new FileNameExtensionFilter("csv files (*.csv)", "csv"))
                .setAcceptAllFileFilterUsed(true)
                .showSaveDialog();

        if (file == null) {
            return;
        }

        // If chosen file exists, use the exact file path
        // If chosen file does not exist and does not have an extension, add it
        if (!file.exists()) {
            if (!file.getAbsolutePath().endsWith(".csv")) {
                file = new File(file.getAbsolutePath() + ".csv");
            }
        }

        try {
            file.createNewFile();

            PrintWriter writer = new PrintWriter(file.getAbsoluteFile(), "UTF-8");
            writer.println("N,X-Coordinate,Z-Coordinate");
            for (int i = 0; i < curve.getNumSegments(); i++) {
                for (int j = 0; j < curve.getSegmentSize(i); ++j) {
                    writer.println(((i+1)*(j+1)) + "," + curve.getSegment(j).get(j).x + "," + curve.getSegment(i).get(j).z);
                }
            }

            writer.close();
        } catch (IOException ex) {
            System.out.println("ERROR writing to a file: " + ex);
        }
    }
    
    private void recomputePrimaryProfile() {
        //Main profile
        CrossSection cs = new CrossSection(getDrawableCuttingPlane(priCuttingPlaneSlot).getPlane());
        getPrimaryDrawableFace().getModel().compute(cs);
        this.primaryCurve = cs.getCrossSectionCurve();

        //Mirror profile
        CrossSection mcs = new CrossSection(getDrawableCuttingPlane(priCuttingPlaneSlot).getMirrorPlane());
        getPrimaryDrawableFace().getModel().compute(mcs);
        this.primaryMirrorCurve = mcs.getCrossSectionCurve();
    }

    private void recomputeSecondaryProfile() {
        //Main profile
        CrossSection cs = new CrossSection(getDrawableCuttingPlane(secCuttingPlaneSlot).getPlane());
        getSecondaryDrawableFace().getModel().compute(cs);
        this.secondaryCurve = cs.getCrossSectionCurve();

        //Mirror profile
        CrossSection mcs = new CrossSection(getDrawableCuttingPlane(secCuttingPlaneSlot).getMirrorPlane());
        getSecondaryDrawableFace().getModel().compute(mcs);
        this.secondaryMirrorCurve = mcs.getCrossSectionCurve();
    }

    private void recomputeProfiles() {
        recomputePrimaryProfile();
        projectCloseFeaturePoints(
                primaryCurve, 
                getCanvas().getPrimaryFace(), 
                getDrawableCuttingPlane(priCuttingPlaneSlot).getPlane());
        projectCloseFeaturePoints(
                primaryMirrorCurve, 
                getCanvas().getPrimaryFace(), 
                getDrawableCuttingPlane(priCuttingPlaneSlot).getPlane());
        controlPanel.getProfileRenderingPanel().setPrimarySegments(this.primaryCurve);
        controlPanel.getProfileRenderingPanel().setPrimaryMirrorSegments(this.primaryMirrorCurve);

        if (getSecondaryDrawableFace() != null) {
            recomputeSecondaryProfile();
            projectCloseFeaturePoints(
                    secondaryCurve, 
                    getCanvas().getSecondaryFace(), 
                    getDrawableCuttingPlane(secCuttingPlaneSlot).getPlane());
            projectCloseFeaturePoints(
                    secondaryMirrorCurve, 
                    getCanvas().getSecondaryFace(), 
                    getDrawableCuttingPlane(secCuttingPlaneSlot).getPlane());
            controlPanel.getProfileRenderingPanel().setSecondarySegments(this.secondaryCurve);
            controlPanel.getProfileRenderingPanel().setSecondaryMirrorSegments(this.secondaryMirrorCurve);
        }
    }

    protected void computeVerticalCuttingPlanes() {
        BBox bbox = getCanvas().getPrimaryFace().getBoundingBox(); 
        Plane plane = new Plane(new Vector3d(1,0,0), bbox.getMidPoint().x);
        DrawableCuttingPlane cuttingPlane = new DrawableCuttingPlane(plane, bbox, true);
        cuttingPlane.setTransparency(0.5f);
        synchronized (this) {
            if (priCuttingPlaneSlot == -1) {
                priCuttingPlaneSlot = getCanvas().getScene().getFreeSlotForOtherDrawables();
            }
            getCanvas().getScene().setOtherDrawable(priCuttingPlaneSlot, cuttingPlane);
        }
        
        if (getSecondaryDrawableFace() != null) {
            bbox = getCanvas().getSecondaryFace().getBoundingBox(); 
            plane = new Plane(new Vector3d(1,0,0), bbox.getMidPoint().x);
            cuttingPlane = new DrawableCuttingPlane(plane, bbox, true);
            cuttingPlane.setTransparency(0.5f);
            synchronized (this) {
                if (secCuttingPlaneSlot == -1) {
                    secCuttingPlaneSlot = getCanvas().getScene().getFreeSlotForOtherDrawables();
                }
                getCanvas().getScene().setOtherDrawable(secCuttingPlaneSlot, new DrawableCuttingPlane(cuttingPlane));
            }
        }
        
        cuttingPlaneFromSymmetry = false;
    }
    
    /**
     * Creates cutting and mirror planes by copying the rectangle from the symmetry plane.
     */
    protected void computeCuttingPlanesFromSymmetry() {
        Plane symmetryPlane = getCanvas().getPrimaryFace().getSymmetryPlane();
        if (symmetryPlane.getNormal().x < 0) { // orient to the right-hand direction
            symmetryPlane = symmetryPlane.flip();
        }
        DrawableCuttingPlane cuttingPlane = new DrawableCuttingPlane(
                new Plane(symmetryPlane), 
                getCanvas().getPrimaryFace().getBoundingBox(),
                false
        );
        cuttingPlane.setTransparency(0.5f);
        synchronized (this) {
            if (priCuttingPlaneSlot == -1) {
                priCuttingPlaneSlot = getCanvas().getScene().getFreeSlotForOtherDrawables();
            }
            getCanvas().getScene().setOtherDrawable(priCuttingPlaneSlot, cuttingPlane);
        }
        recomputePrimaryProfile();
        
        if (getCanvas().getSecondaryFace() != null) {
            symmetryPlane = getCanvas().getSecondaryFace().getSymmetryPlane();
            if (symmetryPlane.getNormal().x < 0) { // orient to the right-hand direction
               symmetryPlane = symmetryPlane.flip();
            }
            cuttingPlane = new DrawableCuttingPlane(
                    new Plane(symmetryPlane), 
                    getCanvas().getSecondaryFace().getBoundingBox(),
                    false
            );
            cuttingPlane.setTransparency(0.5f);
            synchronized (this) {
                if (secCuttingPlaneSlot == -1) {
                    secCuttingPlaneSlot = getCanvas().getScene().getFreeSlotForOtherDrawables();
                }
                getCanvas().getScene().setOtherDrawable(secCuttingPlaneSlot, cuttingPlane);
            }
            recomputeSecondaryProfile();
        }
        cuttingPlaneFromSymmetry = true;
    }

    protected void projectCloseFeaturePoints(CrossSectionCurve curve, HumanFace face, Plane cuttingPlane) {
        if (!face.hasFeaturePoints() || curve.getNumSegments() == 0) {
            return;
        }
        List<Point3d> close = face.getFeaturePoints()
                .stream()
                .filter(fp -> Math.abs(cuttingPlane.getPointDistance(fp.getPosition())) < FEATUE_POINTS_CLOSENESS)
                .map(fp -> fp.getPosition())
                .collect(Collectors.toList());
        curve.projectFeaturePointsToCurve(close);
    }
    
    protected DrawableCuttingPlane getDrawableCuttingPlane(int slot) {
        return (DrawableCuttingPlane) getScene().getOtherDrawable(slot);
    }
    
    protected void showCuttingPlanes(boolean show, boolean showMirrors) {
        if (getDrawableCuttingPlane(priCuttingPlaneSlot) != null) {
            getDrawableCuttingPlane(priCuttingPlaneSlot).show(show);
            getDrawableCuttingPlane(priCuttingPlaneSlot).showMirrorPlane(showMirrors);
        }
        if (getDrawableCuttingPlane(secCuttingPlaneSlot) != null) {
            getDrawableCuttingPlane(secCuttingPlaneSlot).show(show);
            getDrawableCuttingPlane(secCuttingPlaneSlot).showMirrorPlane(showMirrors);
        }
    }
    
    protected double maxExtentX() {
        double max = Double.NEGATIVE_INFINITY;
        
        double planeX = getDrawableCuttingPlane(priCuttingPlaneSlot).getNonShiftedPlane().getPlanePoint().x;
        double bboxX = getCanvas().getPrimaryFace().getBoundingBox().getMaxPoint().x;
        max = Math.max(max, Math.abs(bboxX - planeX));
        bboxX = getCanvas().getPrimaryFace().getBoundingBox().getMinPoint().x;
        max = Math.max(max, Math.abs(bboxX - planeX));
        
        if (getCanvas().getSecondaryFace() != null) {
            planeX = getDrawableCuttingPlane(secCuttingPlaneSlot).getNonShiftedPlane().getPlanePoint().x;
            bboxX = getCanvas().getSecondaryFace().getBoundingBox().getMaxPoint().x;
            max = Math.max(max, Math.abs(bboxX - planeX));
            bboxX = getCanvas().getSecondaryFace().getBoundingBox().getMinPoint().x;
            max = Math.max(max, Math.abs(bboxX - planeX));
        }
        
        return max;
    }
            
}
