package cz.fidentis.analyst.scene;

import com.jogamp.opengl.GL2;
import com.jogamp.opengl.util.texture.Texture;
import com.jogamp.opengl.util.texture.TextureIO;
import cz.fidentis.analyst.mesh.core.MeshFacet;
import cz.fidentis.analyst.mesh.core.MeshModel;
import cz.fidentis.analyst.mesh.core.MeshPoint;
import java.awt.Color;
import java.io.File;
import java.util.List;

/**
 * A drawable triangular mesh, i.e., a mesh model with drawing information like 
 * material, transparency, color, relative transformations in the scene etc. 
 * This class encapsulates rendering state and parameters,
 * 
 * @author Radek Oslejsek
 * @author Richard Pajersky
 * @author Katerina Zarska
 */
public class DrawableMesh extends Drawable {
    
    private final MeshModel model;
    private Texture texture = null;
    
    /**
     * Copy constructor. 
     * 
     * @param mesh Mesh to be copied
     * @throws NullPointException if the input argument is {@code null}
     */
    public DrawableMesh(DrawableMesh mesh) {
        super(mesh);
        this.model = new MeshModel(mesh.getModel());
    }
    
    /**
     * Constructor. 
     * 
     * @param model Drawable mesh model
     * @throws IllegalArgumentException if the model is {@code null}
     */
    public DrawableMesh(MeshModel model) {
        if (model == null) {
            throw new IllegalArgumentException("model is null");
        }
        this.model = model;
    }
    
    /**
     * Constructor. 
     * 
     * @param facet Mesh facet
     * @throws IllegalArgumentException if the model is {@code null}
     */
    public DrawableMesh(MeshFacet facet) {
        if (facet == null) {
            throw new IllegalArgumentException("facet is null");
        }
        this.model = new MeshModel();
        this.model.addFacet(facet);
    }
    
    @Override
    protected void initRendering(GL2 gl) {
        super.initRendering(gl);
        
        // showBackface
        //float[] pos = {0f, 0f, -1f, 0f};
        //gl.glLightfv(GL2.GL_LIGHT1, GL2.GL_POSITION, pos, 0);
        //if (isShowBackface()) {
        //    gl.glLightfv(GL2.GL_LIGHT1, GL2.GL_DIFFUSE, Color.white.getComponents(null), 0);
        //} else {
        //    gl.glLightfv(GL2.GL_LIGHT1, GL2.GL_DIFFUSE, Color.black.getComponents(null), 0);
        //}
    
        // set color of highlights
        Color col = getHighlights();
        float[] highlights = {col.getRed(), col.getGreen(), col.getBlue(), 1};
        gl.glMaterialf(GL2.GL_FRONT_AND_BACK, GL2.GL_SHININESS, 50);  
        gl.glMaterialfv(GL2.GL_FRONT_AND_BACK, GL2.GL_SPECULAR, highlights, 0);
        
        if (getRenderMode() == RenderingMode.TEXTURE && 
                model.hasMaterial() && 
                model.getMaterial().hasTexture() && 
                (texture == null)) {
            
            gl.glTexParameterf(GL2.GL_TEXTURE_2D, GL2.GL_TEXTURE_MAG_FILTER, GL2.GL_NEAREST);
            gl.glTexParameterf(GL2.GL_TEXTURE_2D, GL2.GL_TEXTURE_MIN_FILTER, GL2.GL_NEAREST);
            gl.glTexEnvf(GL2.GL_TEXTURE_ENV, GL2.GL_TEXTURE_ENV_MODE, GL2.GL_REPLACE);
            try{
                texture = TextureIO.newTexture(new File(model.getMaterial().getTexturePath()), true);
                texture.bind(gl);
            } catch (Exception ex){
                ex.printStackTrace();
            }
        }
 
   }
    
    @Override
    protected void renderObject(GL2 gl) {
        if (getRenderMode() == RenderingMode.TEXTURE){
            gl.glEnable(GL2.GL_TEXTURE_2D);
        } else {
            gl.glDisable(GL2.GL_TEXTURE_2D);
        }

        for (MeshFacet facet: getFacets()) {
            gl.glBegin(GL2.GL_TRIANGLES); //vertices are rendered as triangles
        
            // get the normal and tex coords indicies for face i  
            for (int v = 0; v < facet.getCornerTable().getSize(); v++) {            
                MeshPoint mPoint = facet.getVertices().get(facet.getCornerTable().getRow(v).getVertexIndex());
                
                // render the normals
                if(mPoint.getNormal() != null) {
                    gl.glNormal3d(mPoint.getNormal().x, mPoint.getNormal().y, mPoint.getNormal().z);
                }
                
                // render the vertices
                if (getRenderMode() == RenderingMode.TEXTURE && mPoint.getTexCoord() != null) {
                    gl.glTexCoord2d(mPoint.getTexCoord().x, mPoint.getTexCoord().y);
                }
                gl.glVertex3d(mPoint.getPosition().x, mPoint.getPosition().y, mPoint.getPosition().z);
            }
        
            gl.glEnd();
        }
    }
    
    /**
     * Returns list of individual facets.
     * 
     * @return list of individual facets.
     */
    public List<MeshFacet> getFacets() {
        return model.getFacets();
    }
    
    /**
     * @return {@link MeshModel}
     */
    public MeshModel getModel() {
        return this.model;
    }

}
