package cz.fidentis.analyst.visitors.octree;

import cz.fidentis.analyst.mesh.core.MeshFacet;
import cz.fidentis.analyst.mesh.core.MeshPoint;
import cz.fidentis.analyst.octree.Octree;
import cz.fidentis.analyst.octree.OctreeVisitor;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;
import javax.vecmath.Point3d;

/**
 * This visitor goes through all the points in mainFacets and for each point it
 * calculates the distance to the other meshes(octrees) by throwing a ray from
 * that point
 *
 * @author Enkh-Undral EnkhBayar
 */
public class OctreeArrayIntersectionVisitor extends OctreeVisitor {

    /**
     * main mesh facets from which the intersections are calculated
     */
    private final Set<MeshFacet> mainFacets;

    /**
     * center of gravity of the main facet
     */
    private Point3d centerOfGravity;
    private final OctreeArrayIntersectionData data = new OctreeArrayIntersectionData();

    /**
     * Constructor
     *
     * @param mainFacet the main Mesh mainFacet from which to calculate the
     * distances. Must not be {@code null}
     * @throws IllegalArgumentException if some parameter is wrong
     */
    public OctreeArrayIntersectionVisitor(MeshFacet mainFacet) {
        this(new HashSet<>(Collections.singleton(mainFacet)));
        if (mainFacet == null) {
            throw new IllegalArgumentException("mainFacet");
        }
    }

    /**
     * Constructor
     *
     * @param mainFacets the main Mesh mainFacet from which to calculate the
     * distances. Must not be {@code null}
     * @throws IllegalArgumentException if some parameter is wrong
     */
    public OctreeArrayIntersectionVisitor(Set<MeshFacet> mainFacets) {
        this(mainFacets, false);
    }

    /**
     * Constructor
     *
     * @param mainFacets the main Mesh mainFacet from which to calculate the
     * distances. Must not be {@code null}
     * @param useCenterOfGravity if true directional vectors can be changed to
     * (center of gravity -> origin point) base on their angle
     * @throws IllegalArgumentException if some parameter is wrong
     */
    public OctreeArrayIntersectionVisitor(Set<MeshFacet> mainFacets, boolean useCenterOfGravity) {
        if (mainFacets == null || mainFacets.isEmpty()
                || (mainFacets.size() == 1 && mainFacets.contains(null))) {
            throw new IllegalArgumentException("mainFacets");
        }
        this.mainFacets = mainFacets;
        if (useCenterOfGravity) {
            centerOfGravity = new Point3d();
            long size = 0;
            for (MeshFacet facet : mainFacets) {
                size += facet.getNumberOfVertices();
            }
            for (MeshFacet facet : mainFacets) {
                for (MeshPoint meshPoint : facet.getVertices()) {
                    centerOfGravity.x += meshPoint.getPosition().x / size;
                    centerOfGravity.y += meshPoint.getPosition().y / size;
                    centerOfGravity.z += meshPoint.getPosition().z / size;
                }
            }
        }
    }

    /**
     * Getter for data class
     *
     * @return data class of OctreeArrayIntersectionData holding all
     * intersections
     */
    public OctreeArrayIntersectionData getDataClass() {
        return data;
    }

    @Override
    public void visitOctree(Octree octree) {
        if (octree == null) {
            throw new IllegalArgumentException("octree is null");
        }

        for (MeshFacet mainFacet : mainFacets) {
            for (int i = 0; i < mainFacet.getNumberOfVertices(); i++) {
                RayIntersectionVisitor visitor = new RayIntersectionVisitor(mainFacet, i, centerOfGravity);
                octree.accept(visitor);
                if (visitor.getIntersectionsData().getData().isEmpty()) {
                    continue;
                }
                data.addIntersections(mainFacet, i, visitor.getIntersectionsData());
            }
        }
    }
}
