package cz.fidentis.analyst.visitors.mesh.sampling;

import cz.fidentis.analyst.mesh.core.MeshPoint;
import java.util.Collections;
import java.util.List;
import java.util.Random;
import java.util.stream.Collectors;

/**
 * Similar downsampling strategy to the {@link UniformSpaceSampling}.
 * However, a random vertex from the mesh is selected from each non-empty cell instead 
 * of computing a new average point. Therefore, the samples lie on the mesh surface.
 * 
 * @author Radek Oslejsek
 */
public class UniformSurfaceSampling extends UniformSpaceSampling {
    
    /** 
     * Constructor for no point sampling (100 %).
     */
    public UniformSurfaceSampling() {
        this(1.0);
    }
    
    /**
     * Constructor for PERCENTAGE point sampling type.
     * 
     * @param perc Percentage - a value in (0.0, 1.0&gt;
     * @throws IllegalArgumentException if the input parameter is wrong
     */
    public UniformSurfaceSampling(double perc) {
        super(perc);
    }
    
    /**
     * Constructor for MAX_VERTICES point sampling type.
     * 
     * @param max Required number of samples. Must be bigger than zero
     * @throws IllegalArgumentException if the input parameter is wrong
     */
    public UniformSurfaceSampling(int max) {
        super(max);
    }
    
    @Override
    public boolean isBackedByOrigMesh() {
        return true;
    }
    
    @Override
    public List<MeshPoint> getSamples() {
         // no downsampling:
        if (getOrigPoints().size() == getNumDownsampledPoints(getOrigPoints().size())) {
            return Collections.unmodifiableList(getOrigPoints());
        }
        
        // Get a random vertex of the cell:
        Random rand = new Random();
        return createGrid().getNonEmptyCells().stream()
                .map(list -> list.get(rand.nextInt(list.size()))) 
                .collect(Collectors.toList());
    }
    
    @Override
    public String toString() {
        if (getSamplingType() == PointSamplingType.PERCENTAGE) {
            return "uniform surface downsampling to " + (int) (getSamplingLimit() * 100) + "%";
        } else {
            return "uniform surface downsampling to " + (int) getSamplingLimit() + " points";
        }
    }
}
