package cz.fidentis.analyst.visitors.octree;

import cz.fidentis.analyst.mesh.core.MeshFacet;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import javax.vecmath.Point3d;
import javax.vecmath.Vector3d;

/**
 * Class to hold information about intersections between a ray and facets.
 *
 * @author Enkh-Undral EnkhBayar
 */
public class RayIntersectionsData {
    private static final double EPS = 1e-10;

    private Map<MeshFacet, Point3d> data = new HashMap<>();
    private final Point3d origin;
    private Vector3d averageVector = null;

    /**
     * Constructor.
     *
     * @param origin origin point from which the ray starts and intersections
     * are calculated. Must not be {@code null}
     * @throws IllegalArgumentException if some origin is {@code null}
     */
    public RayIntersectionsData(Point3d origin) {
        if (origin == null) {
            throw new IllegalArgumentException("origin is null");
        }
        this.origin = origin;
    }

    /**
     * Getter for origin
     *
     * @return origin point of the ray
     */
    public Point3d getOrigin() {
        return origin;
    }

    /**
     * Adds intersection
     *
     * @param facet facet with which the ray intersects. Must not be {@code null}
     * @param newIntersection new intersection between facet and a point. Must
     * not be {@code null}
     * @throws IllegalArgumentException if some parameter is null
     */
    void addIntersection(MeshFacet facet, Point3d newIntersection) {
        if (facet == null) {
            throw new IllegalArgumentException("facet is null");
        }
        if (newIntersection == null) {
            throw new IllegalArgumentException("newIntersection is null");
        }

        if (data.containsKey(facet)) {
            double newDistance = newIntersection.distance(origin);
            double oldDistance = data.get(facet).distance(origin);
            if (newDistance < oldDistance) {
                data.put(facet, newIntersection);
                averageVector = null;
            }
        } else {
            data.put(facet, newIntersection);
            averageVector = null;
        }
    }

    /**
     * Merges the data of this and other. Resulting data is saved in this and
     * other is not changed.
     *
     * @param other other RayIntersectionData class. Must not be {@code null}.
     * Must have the same origin as this/
     * @throws IllegalArgumentException if some parameter is null
     */
    public void mergeIntersection(RayIntersectionsData other) {
        if (other == null) {
            throw new IllegalArgumentException("other is null");
        }
        if (!other.getOrigin().epsilonEquals(origin, EPS)) {
            throw new IllegalArgumentException("other has the same origin point as this");
        }
        for (var entry : other.getData().entrySet()) {
            addIntersection(entry.getKey(), entry.getValue());
        }
    }

    /**
     * Calculates average vector from intersections.
     *
     * @return average vector from origin of the ray
     */
    public Vector3d getAverageVector() {
        if (averageVector != null) {
            return averageVector;
        }

        Vector3d sumVector = new Vector3d();
        for (Point3d intersection : data.values()) {
            Vector3d tmpVector = new Vector3d(intersection);
            tmpVector.sub(origin);
            sumVector.add(tmpVector);
        }
        averageVector = new Vector3d(sumVector);
        averageVector.scale(1.0 / (data.size() + 1));
        return averageVector;
    }

    /**
     * Getter for data in form of a map. Keys are facets and values are
     * intersections
     *
     * @return data
     */
    Map<MeshFacet, Point3d> getData() {
        return Collections.unmodifiableMap(data);
    }
}