package cz.fidentis.analyst.visitors.octree;

import cz.fidentis.analyst.mesh.core.MeshFacet;
import cz.fidentis.analyst.mesh.core.MeshFacetImpl;
import cz.fidentis.analyst.mesh.core.MeshModel;
import cz.fidentis.analyst.octree.Octree;
import cz.fidentis.analyst.octree.OctreeVisitor;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

/**
 * Visitor capable to create an average face by deforming main face template so
 * that its vertices are in the "average position" with respect to the visited
 * faces. The average position is computed as the average position of
 * intersections of visited faces. Intersection are computed using ray tracing
 * <strong>
 * It is supposed that the inspected faces are already registered
 * (superimposed
 * with the template ).
 * </strong>
 *
 * @author Enkh-Undral EnkhBayar
 */
public class AvgFaceConstructorOctree extends OctreeVisitor {

    private MeshModel avgMeshModel = null;

    private final OctreeArrayIntersectionVisitor visitor;

    /**
     * Constructor.
     *
     * @param templateFacet Mesh facet which is transformed to the averaged mesh.
     *        The original mesh remains unchanged. New mesh is allocated instead.
     * @throws IllegalArgumentException if some parameter is wrong
     */
    public AvgFaceConstructorOctree(MeshFacet templateFacet) {
        this(new HashSet<>(Collections.singleton(templateFacet)));
        if (templateFacet == null) {
            throw new IllegalArgumentException("templateFacet");
        }
    }

    /**
     * Constructor.
     *
     * @param templateFacets Mesh facets that are transformed to the averaged mesh
     *        The original mesh remains unchanged. New mesh is allocated instead.
     * @throws IllegalArgumentException if some parameter is wrong
     */
    public AvgFaceConstructorOctree(Set<MeshFacet> templateFacets) {
        if (templateFacets == null || templateFacets.isEmpty()) {
            throw new IllegalArgumentException("templateFacets");
        }

        visitor = new OctreeArrayIntersectionVisitor(templateFacets);
    }

    /**
     * Constructor.
     *
     * @param templateModel Mesh model which is transformed to the averaged mesh
     *        The original mesh remains unchanged. New mesh is allocated instead.
     * @throws IllegalArgumentException if some parameter is wrong
     */
    public AvgFaceConstructorOctree(MeshModel templateModel) {
        this(new HashSet<>(templateModel.getFacets()));
    }

    @Override
    public void visitOctree(Octree octree) {
        avgMeshModel = null;
        visitor.visitOctree(octree);
    }

    /**
     * Computes and returns the averaged Mesh model
     *
     * @return averaged mesh model
     */
    public MeshModel getAveragedMeshModel() {
        if (avgMeshModel != null) {
            return avgMeshModel;
        }

        avgMeshModel = new MeshModel();
        var result = visitor.getDataClass().getData();
        for (var entry : result.entrySet()) {
            MeshFacet transformedFacet = new MeshFacetImpl(entry.getKey());
            for (var innerEntry : entry.getValue().entrySet()) {
                int index = innerEntry.getKey();
                RayIntersectionsData data = innerEntry.getValue();
                transformedFacet.getVertex(index).getPosition().add(data.getAverageVector());
            }
            avgMeshModel.addFacet(transformedFacet);
        }
        return avgMeshModel;
    }
}
