package cz.fidentis.analyst.face;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Singleton flyweight factory that creates and caches human faces. Faces are
 * stored in the memory until there is enough space in the Java heap. Then they 
 * are cached on disk automatically. The dumping strategy is directed by sub-classes
 * of this abstract class.
 * <p>
 * Dumping faces on disk and their recovery is time consuming (we use serialization
 * of all data structure, i.e., triangular mesh, k-d tree, etc.). It can be faster
 * to re-load the face from OBJ file again and re-compute the data structures from scratch.
 * </p>
 * <p>
 * Currently, listeners registered to {@code HumanFace} and {@code KdTree} 
 * are neither dumped nor reconstructed!
 * </p>
 * 
 * @author Radek Oslejsek
 */
public abstract class HumanFaceCache {
    
    /**
     * Keep at least this portion of the Java heap memory free
     */
    public static final double MIN_FREE_MEMORY = 0.2; // 20%
    
    private final Map<String, HumanFace> inMemoryFaces = new HashMap<>();
    private final Map<String, File> dumpedFaces = new HashMap<>();
    
    /**
     * Private constructor. Use {@link instance} instead to get and access the instance.
     */
    protected HumanFaceCache() {
    }
    
    /**
     * Loads new face. If the face is already loaded, then the existing instance
     * is returned. 
     * 
     * @param file OBJ file with human face.
     * @return Human face or {@code null}
     */
    public HumanFace loadFace(File file) {
        try {
            String faceId = file.getCanonicalPath();
            //String faceId = Long.toHexString(Double.doubleToLongBits(Math.random()));
            
            // In memory face:
            HumanFace face = inMemoryFaces.get(faceId);
            if (face != null) {
                return face;
            }
            
            // Dumped face:
            face = recoverFace(faceId);
            if (face != null) {
                return face;
            }
            
            // New face:
            return storeNewFace(file);
        } catch (IOException|ClassNotFoundException|InterruptedException|ExecutionException ex) {
            Logger.getLogger(HumanFaceCache.class.getName()).log(Level.SEVERE, null, ex);
            return null;
        }
    }
    
    /**
     * Returns a human face. Recovers the face from a dump file if necessary.
     * 
     * @param faceId ID of the face
     * @return Human face or {@code null}
     */
    public HumanFace getFace(String faceId) {
        try {
            // In memory face:
            HumanFace face = inMemoryFaces.get(faceId);
            if (face != null) {
                return face;
            }
            
            // Dumped face:
            face = recoverFace(faceId);
            if (face != null) {
                return face;
            }
            
            // New face:
            return null;
        } catch (IOException|ClassNotFoundException|InterruptedException|ExecutionException ex) {
            Logger.getLogger(HumanFaceCache.class.getName()).log(Level.SEVERE, null, ex);
            return null;
        }
    }
    
    /**
     * Removed the face from the cache.
     * 
     * @param faceId Face ID
     */
    public void removeFace(String faceId) {
        if (inMemoryFaces.remove(faceId) == null) {
            dumpedFaces.remove(faceId);
        }
    }
    
    @Override
    public String toString() {
        StringBuilder ret = new StringBuilder();
        ret.append(formatSize(Runtime.getRuntime().freeMemory())).
                append(" free memory out of ").
                append(formatSize(Runtime.getRuntime().maxMemory()));
        ret.append(". In memory: ").
                append(this.inMemoryFaces.size()).
                append(", dumped: ").
                append(this.dumpedFaces.size());
        return ret.toString();
    }
    
    /**
     * Recovers the face from the dump file.Heap memory is released (some existing 
     * face is dumped) if necessary. 
     * 
     * @param faceId Face ID
     * @return Recovered face or {@code null} if the face was not dumped.
     * @throws IOException on I/O error
     * @throws ClassNotFoundException on I/O error
     * @throws java.lang.InterruptedException
     * @throws java.util.concurrent.ExecutionException
     */
    protected HumanFace recoverFace(String faceId) throws IOException, ClassNotFoundException, InterruptedException, ExecutionException {
        File dumpFile = dumpedFaces.get(faceId);
        if (dumpFile == null) {
            return null;
        }

        // Free memory and recover human face from dump file silultanously:
        ExecutorService executor = Executors.newFixedThreadPool(Runtime.getRuntime().availableProcessors());
        final Future<HumanFace> result = executor.submit(() -> HumanFace.restoreFromFile(dumpFile));
        executor.submit(() -> freeMemory());
        executor.shutdown();
        while (!executor.isTerminated()){}
        HumanFace face = result.get();
        
        inMemoryFaces.put(faceId, face);
        dumpedFaces.remove(faceId);
        return face;
    }
    
    /**
     * Instantiates new face.Heap memory is released (some existing face is dumped) if necessary. 
     * 
     * @param file OBJ file
     * @return Face instance.
     * @throws IOException on I/O error
     * @throws java.lang.InterruptedException
     * @throws java.util.concurrent.ExecutionException
     */
    protected HumanFace storeNewFace(File file) throws IOException, InterruptedException, ExecutionException {
        // Free memory and load human face simultaneously:
        ExecutorService executor = Executors.newFixedThreadPool(Runtime.getRuntime().availableProcessors());
        final Future<HumanFace> result = executor.submit(() -> new HumanFace(file));
        executor.submit(() -> freeMemory());
        executor.shutdown();
        while (!executor.isTerminated()){}
        HumanFace face = result.get();
        inMemoryFaces.put(face.getId(), face);
        return face;
    }
    
    /**
     * Checks and releases the heap, if necessary.
     * 
     * @return true if some existing face has been dumped to free the memory.
     * @throws IOException on I/O error
     */
    protected boolean freeMemory() throws IOException {
        double ratio = (double) Runtime.getRuntime().freeMemory() / Runtime.getRuntime().maxMemory();
        if (ratio > MIN_FREE_MEMORY) {
            return false;
        }
        String faceId = selectFaceForDump();
        if (faceId != null) {
            HumanFace face = inMemoryFaces.remove(faceId);
            dumpedFaces.put(faceId, face.dumpToFile());
            return true;
        }
        return false;
    }
    
    /**
     * Selects a face that will be dumped to file.
     * 
     * @return ID of the face that will be dumped to file.
     */
    protected abstract String selectFaceForDump();
    
    protected static String formatSize(long v) {
        if (v < 1024) {
            return v + " B";
        }
        int z = (63 - Long.numberOfLeadingZeros(v)) / 10;
        return String.format("%.1f %sB", (double)v / (1L << (z*10)), " KMGTPE".charAt(z));
    }
    
    protected Map<String, HumanFace> getInMemoryFaces() {
        return this.inMemoryFaces;
    }
    
    protected Map<String, File> getDumpedFaces() {
        return this.dumpedFaces;
    }
}
