package cz.fidentis.analyst.distance;

import cz.fidentis.analyst.core.ControlPanel;
import cz.fidentis.analyst.core.ControlPanelBuilder;
import cz.fidentis.analyst.feature.FeaturePoint;
import cz.fidentis.analyst.feature.FeaturePointType;
import cz.fidentis.analyst.scene.DrawableFeaturePoints;
import cz.fidentis.analyst.symmetry.SymmetryPanel;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import javax.swing.BorderFactory;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextField;

/**
 * Control panel for Hausdorff distance.
 * 
 * @author Radek Oslejsek
 * @author Daniel Schramm
 */
public class DistancePanel extends ControlPanel {
    
    /*
     * Mandatory design elements
     */
    public static final String ICON = "distance28x28.png";
    public static final String NAME = "Hausdorff distance";
    
    /*
     * External actions
     */
    public static final String ACTION_COMMAND_SHOW_HIDE_HEATMAP = "show-hide heatmap";
    public static final String ACTION_COMMAND_SET_DISTANCE_STRATEGY = "set strategy";
    public static final String ACTION_COMMAND_RELATIVE_ABSOLUTE_DIST = "switch abosulte-relative distance";
    public static final String ACTION_COMMAND_WEIGHTED_DISTANCE = "switch weighted distance on/off";
    public static final String ACTION_COMMAND_FEATURE_POINT_HIGHLIGHT = "highlight feature point with color";
    public static final String ACTION_COMMAND_FEATURE_POINT_RESIZE = "set size of feature point";
    public static final String ACTION_COMMAND_WEIGHTED_DIST_RECOMPUTE = "recompute weighted distance";
    
    /*
     * Configuration of panel-specific GUI elements
     */
    public static final String STRATEGY_POINT_TO_POINT = "Point to point";
    public static final String STRATEGY_POINT_TO_TRIANGLE = "Point to triangle";
    
    private final JPanel featurePointsStats;
    private final JLabel avgHD, maxHD, minHD;
    
    /**
     * Constructor.
     * @param action Action listener
     */
    public DistancePanel(ActionListener action, List<FeaturePoint> featurePoints) {
        this.setName(NAME);
        
        final ControlPanelBuilder builder = new ControlPanelBuilder(this);
        
        builder.addCaptionLine("Computation options:");
        builder.addLine();
        
        builder.addCheckBoxOptionLine(
                null, 
                "Relative distance", 
                false,
                createListener(action, ACTION_COMMAND_RELATIVE_ABSOLUTE_DIST)
        );
        builder.addLine();
        
        builder.addOptionText("Distance strategy");
        builder.addComboBox(
                List.of(STRATEGY_POINT_TO_POINT, STRATEGY_POINT_TO_TRIANGLE),
                createListener(action, ACTION_COMMAND_SET_DISTANCE_STRATEGY)
        );
        builder.addGap();
        builder.addLine();
        
        final JCheckBox weightedDistChBx = builder.addCheckBoxOptionLine(
                null,
                "Weighted Hausdorff distance",
                false,
                createListener(action, ACTION_COMMAND_WEIGHTED_DISTANCE)
        );
        builder.addLine();
        
        final JPanel featurePointsPanel = new JPanel();
        final ControlPanelBuilder fpBuilder = new ControlPanelBuilder(featurePointsPanel);
        final List<JCheckBox> fpCheckBoxes = new ArrayList<>(featurePoints.size());
        final List<JTextField> fpSliderInputs = new ArrayList<>(featurePoints.size());
        for (int i = 0; i < featurePoints.size(); i++) {
            final FeaturePoint featurePoint = featurePoints.get(i);
            
            final JCheckBox checkBox = fpBuilder.addCheckBox(
                    false,
                    createListener(action, ACTION_COMMAND_FEATURE_POINT_HIGHLIGHT, i)
            );
            checkBox.setText(featurePoint.getFeaturePointType().getName());
            fpCheckBoxes.add(checkBox);

            final JTextField sliderInput = fpBuilder.addSliderOptionLine(
                    null,
                    null,
                    100,
                    createListener(action, ACTION_COMMAND_FEATURE_POINT_RESIZE, i)
            );
            sliderInput.setText(ControlPanelBuilder.doubleToStringLocale(DrawableFeaturePoints.DEFAULT_SIZE));
            sliderInput.postActionEvent();
            fpSliderInputs.add(sliderInput);
            
            fpBuilder.addLine();
        }
        builder.addScrollPane(featurePointsPanel)
                .setBorder(BorderFactory.createTitledBorder("Feature points"));
        builder.addLine();
        
        final JButton recomputeButton = builder.addButton(
                "Recompute",
                createListener(action, ACTION_COMMAND_WEIGHTED_DIST_RECOMPUTE)
        );
        recomputeButton.setEnabled(false);
        recomputeButton.addActionListener((ActionEvent ae) -> 
            recomputeButton.setEnabled(false)
        );
        weightedDistChBx.addActionListener((ActionEvent ae) -> {
            if (!weightedDistChBx.isSelected()) {
                recomputeButton.setEnabled(false);
            }
        });
        for (final JCheckBox checkBox: fpCheckBoxes) {
            checkBox.addActionListener((ActionEvent ae) -> {
                if (weightedDistChBx.isSelected()) {
                    recomputeButton.setEnabled(true);
                }
            });
        }
        for (int i = 0; i < fpSliderInputs.size(); i++) {
            final JTextField sliderInput = fpSliderInputs.get(i);
            final JCheckBox sliderCheckBox = fpCheckBoxes.get(i);
            sliderInput.addActionListener((ActionEvent ae) -> {
                if (weightedDistChBx.isSelected() && sliderCheckBox.isSelected()) {
                    recomputeButton.setEnabled(true);
                }
            });
        }
        builder.addLine();
        
        builder.addCaptionLine("Visualization options:");
        builder.addLine();
        builder.addCheckBoxOptionLine(
                null, 
                "Show Hausdorff distance", 
                false,
                createListener(action, ACTION_COMMAND_SHOW_HIDE_HEATMAP)
        ); 
        builder.addLine();
        
        builder.addCaptionLine("Hausdorff distance:");
        builder.addLine();
        
        builder.addOptionText("Average");
        builder.addGap();
        avgHD = builder.addLabelLine("");
        builder.addGap();
        builder.addLine();
        builder.addOptionText("Maximum");
        builder.addGap();
        maxHD = builder.addLabelLine("");
        builder.addGap();
        builder.addLine();
        builder.addOptionText("Minimum");
        builder.addGap();
        minHD = builder.addLabelLine("");
        builder.addGap();
        builder.addLine();
        
        featurePointsStats = new JPanel();
        final JScrollPane fpStatsScrollPanel = builder.addScrollPane(featurePointsStats);
        fpStatsScrollPanel.setBorder(BorderFactory.createTitledBorder("Feature point weights"));
        fpStatsScrollPanel.setVisible(false);
        weightedDistChBx.addActionListener((ActionEvent ae) -> {
            fpStatsScrollPanel.setVisible(weightedDistChBx.isSelected());
        });
        builder.addLine();
        
        builder.addVerticalStrut();
    }
    
    @Override
    public ImageIcon getIcon() {
        return getStaticIcon();
    }
    
    /**
     * Static implementation of the {@link #getIcon()} method.
     * 
     * @return Control panel icon
     */
    public static ImageIcon getStaticIcon() {
        return new ImageIcon(SymmetryPanel.class.getClassLoader().getResource("/" + ICON));
    }
    
    /**
     * Updates GUI elements that display statistical data about the calculated Hausdorff distance.
     * 
     * @param averageDist Average distance
     * @param maxDist Maximum distance
     * @param minDist Minimum distance
     */
    public void updateHausdorffDistanceStats(double averageDist, double maxDist, double minDist) {
        avgHD.setText(Double.toString(averageDist));
        maxHD.setText(Double.toString(maxDist));
        minHD.setText(Double.toString(minDist));
    }
    
    /**
     * Updates GUI elements that display the weights of feature points
     * used to calculate the weighted Hausdorff distance.
     * 
     * @param featurePointWeights Map of feature point types and their weights
     */
    public void updateFeaturePointWeights(Map<FeaturePointType, Double> featurePointWeights) {
        featurePointsStats.removeAll();
        
        final ControlPanelBuilder fpBuilder = new ControlPanelBuilder(featurePointsStats);
        for (final Map.Entry<FeaturePointType, Double> fpStats: featurePointWeights.entrySet()) {
            fpBuilder.addOptionText(fpStats.getKey().getName());
            fpBuilder.addGap();
            
            fpBuilder.addLabelLine(Double.toString(fpStats.getValue()));
            fpBuilder.addGap();
            fpBuilder.addLine();
        }
    }

}
