package cz.fidentis.analyst.symmetry;

import cz.fidentis.analyst.core.ControlPanel;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.Arrays;
import java.util.DoubleSummaryStatistics;
import javax.swing.ImageIcon;
import javax.swing.JOptionPane;

/**
 * Control panel for symmetry plane.
 *
 * @author Natalia Bebjakova
 * @author Radek Oslejsek
 */
public class SymmetryPanel extends ControlPanel  {
    
    /*
     * Handled actions
     */
    public static final String ACTION_COMMAND_RECOMPUTE_FROM_MESH = "Mesh vertices";
    public static final String ACTION_COMMAND_COMPUTE_FROM_FPS = "Feature points";
    public static final String ACTION_COMMAND_POINT_SAMPLING_STRENGTH = "Point sampling strength";
    public static final String ACTION_COMMAND_POINT_SAMPLING_STRATEGY = "Point sampling strategy";
    
    public static final String[] POINT_SAMPLING_STRATEGIES = new String[] {
        "Random Sampling",
        "Mean Curvature",
        "Gaussian Curvature",
        "Max Curvature",
        "Min Curvature",
    };
    
    /*
     * Mandatory design elements
     */
    public static final String ICON = "symmetry28x28.png";
    public static final String NAME = "Symmetry";
    
    /*
     * Configuration of panel-specific GUI elements
     */
    public static final int MAX_SIGNIFICANT_POINTS = 500;
    
    /*
     * Computational state
     */
    private SymmetryConfig config = new SymmetryConfig();

    /**
     * Creates new form SymmetryPanelNew
     */
    public SymmetryPanel(ActionListener action) {
        this.setName(NAME);
        initComponents();
        
        Arrays.stream(POINT_SAMPLING_STRATEGIES).forEach(v -> jComboBox2.addItem(v));
        jComboBox2.setSelectedIndex(0);
        jComboBox2.addActionListener(createListener(action, ACTION_COMMAND_POINT_SAMPLING_STRATEGY));
        
        spinSlider1.initInteger(config.getSignificantPointCount(), 10, MAX_SIGNIFICANT_POINTS, 1);
        spinSlider1.addSpinnerListener((ActionEvent e) -> { 
            config.setSignificantPointCount((Integer) spinSlider1.getValue());
            action.actionPerformed(new ActionEvent(
                    spinSlider1, 
                    ActionEvent.ACTION_PERFORMED, 
                    ACTION_COMMAND_POINT_SAMPLING_STRENGTH)
            ); 
        });
        
        spinSlider2.initDouble(config.getMinCurvRatio(), 0.0, 1.0, 3);
        spinSlider2.addSpinnerListener((ActionEvent e) -> { 
            config.setMinCurvRatio((Double) spinSlider2.getValue());
        });
        
        spinSlider3.initDouble(config.getMinAngleCos(), 0.0, 1.0, 3);
        spinSlider3.addSpinnerListener((ActionEvent e) -> { 
            config.setMinAngleCos((Double) spinSlider3.getValue());
        });
        
        spinSlider4.initDouble(config.getMinNormAngleCos(), 0.0, 1.0, 3);
        spinSlider4.addSpinnerListener((ActionEvent e) -> { 
            config.setMinNormAngleCos((Double) spinSlider4.getValue());
        });
        
        spinSlider5.initDouble(config.getMaxRelDistance(), 0.0, 1.0, 2);
        spinSlider5.addSpinnerListener((ActionEvent e) -> { 
            config.setMaxRelDistance((Double) spinSlider5.getValue());
        });

        jButton1.addActionListener((ActionEvent e) -> {
            config = new SymmetryConfig();
            spinSlider1.setValue(config.getSignificantPointCount());
            spinSlider2.setValue(config.getMinCurvRatio());
            spinSlider3.setValue(config.getMinAngleCos());
            spinSlider4.setValue(config.getMinNormAngleCos());
            spinSlider5.setValue(config.getMaxRelDistance());
            //setConfig(new SymmetryConfig());
        });
        
        jButton2.addActionListener((ActionEvent e) -> { 
            action.actionPerformed(new ActionEvent( // recompute
                    e.getSource(), 
                    ActionEvent.ACTION_PERFORMED,
                    jComboBox1.getSelectedItem().toString()
            )); 
        });
        
        jButtonInfo1.addActionListener((ActionEvent e) -> { 
            this.showSignPointsHelp();
        });
        
        jButtonInfo2.addActionListener((ActionEvent e) -> { 
            this.showMinCurvHelp();
        });
        
        jButtonInfo3.addActionListener((ActionEvent e) -> { 
            this.showMinAngleCosHelp();
        });
        
        jButtonInfo4.addActionListener((ActionEvent e) -> { 
            this.showNormalAngleHelp();
        });
        
        jComboBox1.addItem(SymmetryPanel.ACTION_COMMAND_RECOMPUTE_FROM_MESH);
        //jComboBox1.addItem(SymmetryPanel.ACTION_COMMAND_COMPUTE_FROM_FPS);
        jComboBox1.setSelectedIndex(0);
    }
    
    @Override
    public ImageIcon getIcon() {
        return getStaticIcon();
    }
    
    /**
     * Return the number of point samples
     * @return the number of point samples
     */
    public int getPointSamplingStrength() {
        return (Integer) spinSlider1.getValue();
    }
    
    /**
     * Return selected point sampling strategy
     * @return selected point sampling strategy
     */
    public String getPointSamplingStrategy() {
        return POINT_SAMPLING_STRATEGIES[jComboBox2.getSelectedIndex()];
    }
    
    /**
     * Returns symmetry plane configuration
     * @return symmetry plane configuration
     */
    public SymmetryConfig getSymmetryConfig() {
        return this.config;
    }
    
    /**
     * Static implementation of the {@link #getIcon()} method.
     * 
     * @return Control panel icon
     */
    public static ImageIcon getStaticIcon() {
        return new ImageIcon(SymmetryPanel.class.getClassLoader().getResource("/" + ICON));
    }

    /**
     * Enables/disables the selection for the computation of the symmetry from feature points.
     * @param on {@code true} = enable
     */
    public void setComputeFromFPs(boolean on) {
        if (on && jComboBox1.getItemCount() == 1) {
            jComboBox1.addItem(SymmetryPanel.ACTION_COMMAND_COMPUTE_FROM_FPS);
        }
        
        if (!on && jComboBox1.getItemCount() == 2) {
            jComboBox1.removeItemAt(1);
        }
    }
    
    /**
     * Updates GUI elements that display statistical data about the calculated Hausdorff distance.
     * 
     * @param stats Statistical data of the ordinary Hausdorff distance
     * @param first First face
     */
    public void updateHausdorffDistanceStats(DoubleSummaryStatistics stats, boolean first) {
        if (first) {
            jTextField1.setText(String.format("%.2f", stats.getAverage()));
        } else {
            jTextField2.setText(String.format("%.2f", stats.getAverage()));
        }
    }
    
    private void showSignPointsHelp() {
        JOptionPane.showMessageDialog(this,
                "Entered number represents amount of points of the mesh that are taken into account" + System.lineSeparator()
                        + "while counting the plane of approximate symmetry." + System.lineSeparator() 
                        + System.lineSeparator()
                        + "Higher number → longer calculation, possibly more accurate result." + System.lineSeparator()
                        + "Lower number → shorter calculation, possibly less accurate result.", 
                "Significant points",
                0, 
                new ImageIcon(getClass().getResource("/points.png"))
        );
                
    }
     
    private void showMinAngleCosHelp() {
        JOptionPane.showMessageDialog(this,
                "Entered number represents how large the angle between normal vector of candidate plane and the vector" + System.lineSeparator()
                        + "of two vertices can be to take into account these vertices while counting the approximate symmetry."  + System.lineSeparator()
                        + System.lineSeparator()
                        + "Higher number → fewer pairs of vertices satisfy the criterion → shorter calculation, possibly less accurate result." + System.lineSeparator()
                        + "Lower number → more pairs of vertices satisfy the criterion → longer calculation, possibly more accurate result.",
                "Minimum angle",
                0, 
                new ImageIcon(getClass().getResource("/angle.png"))
        );
    }
    
    private void showNormalAngleHelp() {                                             
        JOptionPane.showMessageDialog(this,
                "Entered number represents how large the angle between normal vector of candidate plane and vector" + System.lineSeparator()
                        + "from subtraction of normal vectors of two vertices can be to take into account these vertices while counting the approximate symmetry." + System.lineSeparator()
                         + System.lineSeparator()
                        + "Higher number → fewer pairs of vertices satisfy the criterion → shorter calculation, possibly less accurate result." + System.lineSeparator()
                        + "Lower number → more pairs of vertices satisfy the criterion → longer calculation, possibly more accurate result.",
                "Minimum normal angle",
                0, 
                new ImageIcon(getClass().getResource("/angle.png"))
        );
    }   
    
    private void showMinCurvHelp() {                                         
        JOptionPane.showMessageDialog(this,
                "Entered number represents how similar the curvature in two vertices must be" + System.lineSeparator()
                        + "to take into account these vertices while counting the plane of approximate symmetry." + System.lineSeparator()
                        + "The higher the number is the more similar they must be." + System.lineSeparator()
                        + System.lineSeparator()
                        + "Higher number → fewer pairs of vertices satisfy the criterion → shorter calculation, possibly less accurate result." + System.lineSeparator()
                        + "Lower number → more pairs of vertices satisfy the criterion → longer calculation, possibly more accurate result.",
                "Minimum curvature ratio",
                0, 
                new ImageIcon(getClass().getResource("/curvature.png"))
        );
    }                                        

    private void showRelDistHelp() {
        JOptionPane.showMessageDialog(this,
                "Entered number represents how far middle point of two vertices can be from candidate plane of symmetry" + System.lineSeparator()
                        + "to give this plane vote. Plane with highest number of votes is plane of approximate symmetry." + System.lineSeparator()
                        + System.lineSeparator()
                        + "Higher number → more pairs of vertices satisfy the criterion → longer calculation, possibly more accurate result." + System.lineSeparator()
                        + "Lower number → fewer pairs of vertices satisfy the criterion → shorter calculation, possibly less accurate result.",
                "Maximum relative distance from plane",
                0, 
                new ImageIcon(getClass().getResource("/distance.png"))
        );
    }
    
    /**
     * This method is called from within the constructor to initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is always
     * regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {

        jPanel1 = new javax.swing.JPanel();
        jLabel1 = new javax.swing.JLabel();
        jLabel2 = new javax.swing.JLabel();
        jLabel3 = new javax.swing.JLabel();
        jLabel4 = new javax.swing.JLabel();
        jLabel5 = new javax.swing.JLabel();
        jLabel6 = new javax.swing.JLabel();
        jCheckBox1 = new javax.swing.JCheckBox();
        jButtonInfo1 = new javax.swing.JButton();
        jButtonInfo2 = new javax.swing.JButton();
        jButtonInfo4 = new javax.swing.JButton();
        jButtonInfo3 = new javax.swing.JButton();
        jButton1 = new javax.swing.JButton();
        jLabel9 = new javax.swing.JLabel();
        jComboBox2 = new javax.swing.JComboBox<>();
        spinSlider1 = new cz.fidentis.analyst.core.SpinSlider();
        spinSlider2 = new cz.fidentis.analyst.core.SpinSlider();
        spinSlider3 = new cz.fidentis.analyst.core.SpinSlider();
        spinSlider4 = new cz.fidentis.analyst.core.SpinSlider();
        spinSlider5 = new cz.fidentis.analyst.core.SpinSlider();
        jPanel2 = new javax.swing.JPanel();
        jTextField1 = new javax.swing.JTextField();
        jLabel7 = new javax.swing.JLabel();
        jLabel8 = new javax.swing.JLabel();
        jTextField2 = new javax.swing.JTextField();
        jPanel3 = new javax.swing.JPanel();
        jComboBox1 = new javax.swing.JComboBox<>();
        jButton2 = new javax.swing.JButton();

        jPanel1.setBorder(javax.swing.BorderFactory.createTitledBorder(null, org.openide.util.NbBundle.getMessage(SymmetryPanel.class, "SymmetryPanel.jPanel1.border.title_1"), javax.swing.border.TitledBorder.DEFAULT_JUSTIFICATION, javax.swing.border.TitledBorder.DEFAULT_POSITION, new java.awt.Font("Dialog", 1, 14))); // NOI18N

        org.openide.awt.Mnemonics.setLocalizedText(jLabel1, org.openide.util.NbBundle.getMessage(SymmetryPanel.class, "SymmetryPanel.jLabel1.text_1")); // NOI18N

        org.openide.awt.Mnemonics.setLocalizedText(jLabel2, org.openide.util.NbBundle.getMessage(SymmetryPanel.class, "SymmetryPanel.jLabel2.text_1")); // NOI18N

        org.openide.awt.Mnemonics.setLocalizedText(jLabel3, org.openide.util.NbBundle.getMessage(SymmetryPanel.class, "SymmetryPanel.jLabel3.text_1")); // NOI18N

        org.openide.awt.Mnemonics.setLocalizedText(jLabel4, org.openide.util.NbBundle.getMessage(SymmetryPanel.class, "SymmetryPanel.jLabel4.text_1")); // NOI18N

        org.openide.awt.Mnemonics.setLocalizedText(jLabel5, org.openide.util.NbBundle.getMessage(SymmetryPanel.class, "SymmetryPanel.jLabel5.text_1")); // NOI18N

        org.openide.awt.Mnemonics.setLocalizedText(jLabel6, org.openide.util.NbBundle.getMessage(SymmetryPanel.class, "SymmetryPanel.jLabel6.text_1")); // NOI18N

        jCheckBox1.setSelected(true);
        org.openide.awt.Mnemonics.setLocalizedText(jCheckBox1, org.openide.util.NbBundle.getMessage(SymmetryPanel.class, "SymmetryPanel.jCheckBox1.text")); // NOI18N

        jButtonInfo1.setIcon(new javax.swing.ImageIcon(getClass().getResource("/info.png"))); // NOI18N
        org.openide.awt.Mnemonics.setLocalizedText(jButtonInfo1, org.openide.util.NbBundle.getMessage(SymmetryPanel.class, "SymmetryPanel.jButtonInfo1.text")); // NOI18N
        jButtonInfo1.setBorderPainted(false);

        jButtonInfo2.setIcon(new javax.swing.ImageIcon(getClass().getResource("/info.png"))); // NOI18N
        org.openide.awt.Mnemonics.setLocalizedText(jButtonInfo2, org.openide.util.NbBundle.getMessage(SymmetryPanel.class, "SymmetryPanel.jButtonInfo2.text")); // NOI18N
        jButtonInfo2.setBorderPainted(false);

        jButtonInfo4.setIcon(new javax.swing.ImageIcon(getClass().getResource("/info.png"))); // NOI18N
        org.openide.awt.Mnemonics.setLocalizedText(jButtonInfo4, org.openide.util.NbBundle.getMessage(SymmetryPanel.class, "SymmetryPanel.jButtonInfo4.text")); // NOI18N
        jButtonInfo4.setBorderPainted(false);

        jButtonInfo3.setIcon(new javax.swing.ImageIcon(getClass().getResource("/info.png"))); // NOI18N
        org.openide.awt.Mnemonics.setLocalizedText(jButtonInfo3, org.openide.util.NbBundle.getMessage(SymmetryPanel.class, "SymmetryPanel.jButtonInfo3.text")); // NOI18N
        jButtonInfo3.setBorderPainted(false);
        jButtonInfo3.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                jButtonInfo3ActionPerformed(evt);
            }
        });

        org.openide.awt.Mnemonics.setLocalizedText(jButton1, org.openide.util.NbBundle.getMessage(SymmetryPanel.class, "SymmetryPanel.jButton1.text")); // NOI18N

        org.openide.awt.Mnemonics.setLocalizedText(jLabel9, org.openide.util.NbBundle.getMessage(SymmetryPanel.class, "SymmetryPanel.jLabel9.text")); // NOI18N

        javax.swing.GroupLayout jPanel1Layout = new javax.swing.GroupLayout(jPanel1);
        jPanel1.setLayout(jPanel1Layout);
        jPanel1Layout.setHorizontalGroup(
            jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(jPanel1Layout.createSequentialGroup()
                .addContainerGap()
                .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addComponent(jLabel9, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                    .addGroup(jPanel1Layout.createSequentialGroup()
                        .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                            .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.TRAILING, false)
                                .addComponent(jLabel4, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                                .addComponent(jLabel3))
                            .addComponent(jLabel2, javax.swing.GroupLayout.PREFERRED_SIZE, 165, javax.swing.GroupLayout.PREFERRED_SIZE)
                            .addComponent(jLabel1)
                            .addComponent(jLabel5, javax.swing.GroupLayout.PREFERRED_SIZE, 143, javax.swing.GroupLayout.PREFERRED_SIZE)
                            .addComponent(jLabel6, javax.swing.GroupLayout.PREFERRED_SIZE, 143, javax.swing.GroupLayout.PREFERRED_SIZE))
                        .addGap(0, 12, Short.MAX_VALUE)))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addComponent(jComboBox2, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addGroup(jPanel1Layout.createSequentialGroup()
                        .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.TRAILING, false)
                            .addGroup(jPanel1Layout.createSequentialGroup()
                                .addComponent(jCheckBox1)
                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                                .addComponent(jButton1))
                            .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                                .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                                    .addComponent(spinSlider2, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                                    .addComponent(spinSlider3, javax.swing.GroupLayout.Alignment.TRAILING, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                                .addComponent(spinSlider4, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                                .addComponent(spinSlider5, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                                .addComponent(spinSlider1, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)))
                        .addGap(18, 18, 18)
                        .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                            .addComponent(jButtonInfo1)
                            .addComponent(jButtonInfo2)
                            .addComponent(jButtonInfo4)
                            .addComponent(jButtonInfo3))))
                .addContainerGap(javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
        );
        jPanel1Layout.setVerticalGroup(
            jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(jPanel1Layout.createSequentialGroup()
                .addContainerGap()
                .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(jLabel9)
                    .addComponent(jComboBox2, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addGap(18, 18, 18)
                .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addComponent(jButtonInfo1)
                    .addComponent(spinSlider1, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addGroup(jPanel1Layout.createSequentialGroup()
                        .addGap(8, 8, 8)
                        .addComponent(jLabel1)))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addGroup(jPanel1Layout.createSequentialGroup()
                        .addGap(8, 8, 8)
                        .addComponent(jLabel2)
                        .addGap(26, 26, 26)
                        .addComponent(jLabel3)
                        .addGap(26, 26, 26)
                        .addComponent(jLabel4)
                        .addGap(28, 28, 28)
                        .addComponent(jLabel5))
                    .addGroup(jPanel1Layout.createSequentialGroup()
                        .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                            .addComponent(spinSlider2, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                            .addComponent(jButtonInfo2))
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                            .addComponent(spinSlider3, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                            .addComponent(jButtonInfo3))
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                            .addComponent(jButtonInfo4)
                            .addComponent(spinSlider4, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(spinSlider5, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)))
                .addGap(24, 24, 24)
                .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addComponent(jButton1)
                    .addGroup(jPanel1Layout.createSequentialGroup()
                        .addGap(5, 5, 5)
                        .addComponent(jLabel6))
                    .addComponent(jCheckBox1, javax.swing.GroupLayout.Alignment.TRAILING))
                .addContainerGap(26, Short.MAX_VALUE))
        );

        jPanel2.setBorder(javax.swing.BorderFactory.createTitledBorder(null, org.openide.util.NbBundle.getMessage(SymmetryPanel.class, "SymmetryPanel.jPanel2.border.title"), javax.swing.border.TitledBorder.DEFAULT_JUSTIFICATION, javax.swing.border.TitledBorder.DEFAULT_POSITION, new java.awt.Font("Dialog", 1, 12))); // NOI18N

        jTextField1.setText(org.openide.util.NbBundle.getMessage(SymmetryPanel.class, "SymmetryPanel.jTextField1.text")); // NOI18N

        org.openide.awt.Mnemonics.setLocalizedText(jLabel7, org.openide.util.NbBundle.getMessage(SymmetryPanel.class, "SymmetryPanel.jLabel7.text")); // NOI18N

        org.openide.awt.Mnemonics.setLocalizedText(jLabel8, org.openide.util.NbBundle.getMessage(SymmetryPanel.class, "SymmetryPanel.jLabel8.text")); // NOI18N

        jTextField2.setText(org.openide.util.NbBundle.getMessage(SymmetryPanel.class, "SymmetryPanel.jTextField2.text")); // NOI18N

        javax.swing.GroupLayout jPanel2Layout = new javax.swing.GroupLayout(jPanel2);
        jPanel2.setLayout(jPanel2Layout);
        jPanel2Layout.setHorizontalGroup(
            jPanel2Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(javax.swing.GroupLayout.Alignment.TRAILING, jPanel2Layout.createSequentialGroup()
                .addContainerGap()
                .addComponent(jLabel7)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addComponent(jTextField1, javax.swing.GroupLayout.PREFERRED_SIZE, 55, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addComponent(jLabel8)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addComponent(jTextField2, javax.swing.GroupLayout.PREFERRED_SIZE, 55, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addContainerGap(21, Short.MAX_VALUE))
        );
        jPanel2Layout.setVerticalGroup(
            jPanel2Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(jPanel2Layout.createSequentialGroup()
                .addContainerGap()
                .addGroup(jPanel2Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(jTextField1, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(jLabel7)
                    .addComponent(jLabel8)
                    .addComponent(jTextField2, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addContainerGap(javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
        );

        jPanel3.setBorder(javax.swing.BorderFactory.createTitledBorder(null, org.openide.util.NbBundle.getMessage(SymmetryPanel.class, "SymmetryPanel.jPanel3.border.title"), javax.swing.border.TitledBorder.DEFAULT_JUSTIFICATION, javax.swing.border.TitledBorder.DEFAULT_POSITION, new java.awt.Font("Dialog", 1, 12))); // NOI18N

        jButton2.setFont(new java.awt.Font("Ubuntu", 1, 15)); // NOI18N
        org.openide.awt.Mnemonics.setLocalizedText(jButton2, org.openide.util.NbBundle.getMessage(SymmetryPanel.class, "SymmetryPanel.jButton2.text")); // NOI18N
        jButton2.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                jButton2ActionPerformed(evt);
            }
        });

        javax.swing.GroupLayout jPanel3Layout = new javax.swing.GroupLayout(jPanel3);
        jPanel3.setLayout(jPanel3Layout);
        jPanel3Layout.setHorizontalGroup(
            jPanel3Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(jPanel3Layout.createSequentialGroup()
                .addContainerGap()
                .addComponent(jComboBox1, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addGap(18, 18, 18)
                .addComponent(jButton2)
                .addContainerGap(javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
        );
        jPanel3Layout.setVerticalGroup(
            jPanel3Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(jPanel3Layout.createSequentialGroup()
                .addContainerGap()
                .addGroup(jPanel3Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(jComboBox1, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(jButton2))
                .addContainerGap(javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
        );

        javax.swing.GroupLayout layout = new javax.swing.GroupLayout(this);
        this.setLayout(layout);
        layout.setHorizontalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(layout.createSequentialGroup()
                .addContainerGap()
                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.TRAILING, false)
                    .addComponent(jPanel3, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                    .addComponent(jPanel1, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                    .addComponent(jPanel2, javax.swing.GroupLayout.Alignment.LEADING, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addContainerGap(javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
        );
        layout.setVerticalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(layout.createSequentialGroup()
                .addContainerGap()
                .addComponent(jPanel3, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addComponent(jPanel1, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addComponent(jPanel2, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addContainerGap(98, Short.MAX_VALUE))
        );
    }// </editor-fold>//GEN-END:initComponents

    private void jButton2ActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_jButton2ActionPerformed
        // TODO add your handling code here:
    }//GEN-LAST:event_jButton2ActionPerformed

    private void jButtonInfo3ActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_jButtonInfo3ActionPerformed
        // TODO add your handling code here:
    }//GEN-LAST:event_jButtonInfo3ActionPerformed


    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JButton jButton1;
    private javax.swing.JButton jButton2;
    private javax.swing.JButton jButtonInfo1;
    private javax.swing.JButton jButtonInfo2;
    private javax.swing.JButton jButtonInfo3;
    private javax.swing.JButton jButtonInfo4;
    private javax.swing.JCheckBox jCheckBox1;
    private javax.swing.JComboBox<String> jComboBox1;
    private javax.swing.JComboBox<String> jComboBox2;
    private javax.swing.JLabel jLabel1;
    private javax.swing.JLabel jLabel2;
    private javax.swing.JLabel jLabel3;
    private javax.swing.JLabel jLabel4;
    private javax.swing.JLabel jLabel5;
    private javax.swing.JLabel jLabel6;
    private javax.swing.JLabel jLabel7;
    private javax.swing.JLabel jLabel8;
    private javax.swing.JLabel jLabel9;
    private javax.swing.JPanel jPanel1;
    private javax.swing.JPanel jPanel2;
    private javax.swing.JPanel jPanel3;
    private javax.swing.JTextField jTextField1;
    private javax.swing.JTextField jTextField2;
    private cz.fidentis.analyst.core.SpinSlider spinSlider1;
    private cz.fidentis.analyst.core.SpinSlider spinSlider2;
    private cz.fidentis.analyst.core.SpinSlider spinSlider3;
    private cz.fidentis.analyst.core.SpinSlider spinSlider4;
    private cz.fidentis.analyst.core.SpinSlider spinSlider5;
    // End of variables declaration//GEN-END:variables
}
