package cz.fidentis.analyst.symmetry;

import cz.fidentis.analyst.Logger;
import cz.fidentis.analyst.canvas.Canvas;
import cz.fidentis.analyst.core.ControlPanelAction;
import cz.fidentis.analyst.core.SpinSlider;
import cz.fidentis.analyst.face.HumanFace;
import cz.fidentis.analyst.face.events.HumanFaceEvent;
import cz.fidentis.analyst.face.events.HumanFaceListener;
import cz.fidentis.analyst.face.events.SymmetryPlaneChangedEvent;
import cz.fidentis.analyst.feature.FeaturePoint;
import cz.fidentis.analyst.mesh.core.MeshFacet;
import cz.fidentis.analyst.mesh.core.MeshModel;
import cz.fidentis.analyst.scene.DrawableFace;
import cz.fidentis.analyst.scene.DrawablePointCloud;
import cz.fidentis.analyst.visitors.mesh.Curvature;
import cz.fidentis.analyst.visitors.mesh.HausdorffDistance;
import cz.fidentis.analyst.visitors.mesh.HausdorffDistance.Strategy;
import cz.fidentis.analyst.visitors.mesh.sampling.CurvatureSampling;
import cz.fidentis.analyst.visitors.mesh.sampling.NoSampling;
import cz.fidentis.analyst.visitors.mesh.sampling.PointSampling;
import cz.fidentis.analyst.visitors.mesh.sampling.RandomSampling;

import java.awt.event.ActionEvent;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.swing.JTabbedPane;
import javax.vecmath.Point3d;
import javax.vecmath.Vector3d;

/**
 * Action listener for the manipulation with the symmetry plane.
 * 
 * @author Radek Oslejsek
 */
public class SymmetryAction extends ControlPanelAction implements HumanFaceListener  {

    private final SymmetryPanel controlPanel;
    private final JTabbedPane topControlPanel;
    private int sPlane = 0; // 0 = none, 1 = from mesh, 2 = from FPs
    
    private int primCloudSlot = -1;
    //private int secCloudSlot = -1;

    /**
     * Constructor.
     * 
     * @param canvas OpenGL canvas
     * @param topControlPanel Top component for placing control panels
     */
    public SymmetryAction(Canvas canvas, JTabbedPane topControlPanel) {
        super(canvas, topControlPanel);
        this.topControlPanel = topControlPanel;
        //this.controlPanel = new SymmetryPanel(this);
        this.controlPanel = new SymmetryPanel(this);
        
        controlPanel.setComputeFromFPs(
                (getPrimaryFeaturePoints() != null && !getPrimaryFeaturePoints().getFeaturePoints().isEmpty()) || 
                        (getSecondaryFeaturePoints() != null && !getSecondaryFeaturePoints().getFeaturePoints().isEmpty())
        );

        // Place control panel to the topControlPanel
        this.topControlPanel.addTab(controlPanel.getName(), controlPanel.getIcon(), controlPanel);
        this.topControlPanel.addChangeListener(e -> {
            // If the symmetry panel is focused...
            if (((JTabbedPane) e.getSource()).getSelectedComponent() instanceof SymmetryPanel) {
                // show point cloud:
                if (getCanvas().getSecondaryFace() == null) { // single face analysis
                    primCloudSlot = drawPointSamples(
                            getCanvas().getScene().getPrimaryFaceSlot(), 
                            primCloudSlot, 
                            controlPanel.getPointSamplingStrength()
                    );
                }
                //getCanvas().getScene().setDefaultColors();
                //getCanvas().getScene().showSymmetryPlane(getCanvas().getScene().getPrimaryFaceSlot(), true);
                //getCanvas().getScene().showSymmetryPlane(getCanvas().getScene().getSecondaryFaceSlot(), true);
            } else { 
                // hide point cloud:
                getScene().setOtherDrawable(primCloudSlot, null);
                //getScene().setOtherDrawable(secCloudSlot, null);
                primCloudSlot = -1;
                //secCloudSlot = -1;
                //getCanvas().getScene().showSymmetryPlane(getCanvas().getScene().getPrimaryFaceSlot(), false);
                //getCanvas().getScene().showSymmetryPlane(getCanvas().getScene().getSecondaryFaceSlot(), false);
            }
        });

        // Be informed about changes in faces perfomed by other GUI elements
        getPrimaryDrawableFace().getHumanFace().registerListener(this);
        if (getSecondaryDrawableFace() != null) {
            getSecondaryDrawableFace().getHumanFace().registerListener(this);
        }
    }
    
    @Override
    public void actionPerformed(ActionEvent ae) {
        String action = ae.getActionCommand();
        
        switch (action) {
            case SymmetryPanel.ACTION_COMMAND_RECOMPUTE_FROM_MESH: 
                sPlane = 1;
                recomputeFromMesh(getCanvas().getScene().getPrimaryFaceSlot());
                recomputeFromMesh(getCanvas().getScene().getSecondaryFaceSlot());
                break; 
            case SymmetryPanel.ACTION_COMMAND_COMPUTE_FROM_FPS: 
                sPlane = 2;
                recomputeFromFeaturePoints(getCanvas().getScene().getPrimaryFaceSlot());
                recomputeFromFeaturePoints(getCanvas().getScene().getSecondaryFaceSlot());
                break;
            case SymmetryPanel.ACTION_COMMAND_POINT_SAMPLING_STRENGTH:
                int numSamples = (Integer) ((SpinSlider) ae.getSource()).getValue();
                if (getCanvas().getSecondaryFace() == null) { // single face analysis
                    primCloudSlot = drawPointSamples(getCanvas().getScene().getPrimaryFaceSlot(), primCloudSlot, numSamples);
                }
                //secCloudSlot = drawPointSamples(getCanvas().getScene().getSecondaryFaceSlot(), secCloudSlot, numSamples);
                break;
            case SymmetryPanel.ACTION_COMMAND_POINT_SAMPLING_STRATEGY:
                if (getCanvas().getSecondaryFace() == null) { // single face analysis
                    primCloudSlot = drawPointSamples(getCanvas().getScene().getPrimaryFaceSlot(), primCloudSlot, controlPanel.getPointSamplingStrength());
                }
                //secCloudSlot = drawPointSamples(getCanvas().getScene().getSecondaryFaceSlot(), secCloudSlot, controlPanel.getPointSamplingStrength());
                break;
            default:
                // do nothing
        }
        renderScene();
    }
    
    @Override
    public void acceptEvent(HumanFaceEvent event) {
        /* symmetry plane is transformed together with the mesh
        if (event instanceof MeshChangedEvent) { // recompute symmetry plane, if necessary
            switch (sPlane) {
                case 0: // none
                    return;
                case 1: // from mesh
                    recomputeFromMesh(event.getFace().equals(getPrimaryDrawableFace().getHumanFace()) 
                            ? getCanvas().getScene().getPrimaryFaceSlot()
                            : getCanvas().getScene().getSecondaryFaceSlot());
                    break;
                case 2: // from FPs
                    recomputeFromFeaturePoints(event.getFace().equals(getPrimaryDrawableFace().getHumanFace())
                            ? getCanvas().getScene().getPrimaryFaceSlot() 
                            : getCanvas().getScene().getSecondaryFaceSlot());
                    break;
                default:
            }
        }
        */
        if (event instanceof SymmetryPlaneChangedEvent) {
            updatePrecision(event.getFace());
        }
    }
    
    protected void recomputeFromMesh(int faceSlot) {
        HumanFace face = getCanvas().getHumanFace(faceSlot);
        if (face != null) {
            Logger log = Logger.measureTime();
            SymmetryEstimator estimator = new SymmetryEstimator(
                    controlPanel.getSymmetryConfig(), 
                    getSamplingStrategy(controlPanel.getPointSamplingStrength(), face.getCurvature())
            );
            face.getMeshModel().compute(estimator);
            face.setSymmetryPlane(estimator.getSymmetryPlane());
            log.printDuration("Symmetry plane estimation from mesh for " + face.getShortName());
            setDrawablePlane(face, faceSlot);
            face.announceEvent(new SymmetryPlaneChangedEvent(face, "", this));
        }
    }
    
    protected void recomputeFromFeaturePoints(int index) {
        HumanFace face = getCanvas().getHumanFace(index);
        if (face == null) {
            return;
        }
        
        Map<String, Point3d> processFP = new HashMap<>();
        List<Plane> planes = new ArrayList<>();
        
        for (FeaturePoint fp: face.getFeaturePoints()) {
            String code = fp.getFeaturePointType().getCode();
            String pairCode;
            if (code.endsWith("_R")) {
                pairCode = code.replaceAll("_R", "_L");
            } else if (code.endsWith("_L")) {
                pairCode = code.replaceAll("_L", "_R");
            } else {
                continue;
            }
            
            if (!processFP.containsKey(pairCode)) {
                processFP.put(code, fp.getPosition());
                continue;
            }
            
            Vector3d normal = new Vector3d(processFP.remove(pairCode));
            Vector3d center = new Vector3d(normal);
                
            normal.sub(fp.getPosition());
            normal.normalize();
                    
            center.add(fp.getPosition());
            center.scale(0.5);
                    
            planes.add(new Plane(normal, normal.dot(center)));
        }
        
        face.setSymmetryPlane(new Plane(planes)); // creates an average plane
        setDrawablePlane(face, index);
        face.announceEvent(new SymmetryPlaneChangedEvent(face, "", this));
    }
    
    protected void setDrawablePlane(HumanFace face, int index) {
        getCanvas().getScene().setDrawableSymmetryPlane(index, face);
        getCanvas().getScene().getDrawableSymmetryPlane(index).setTransparency(0.5f);
        getCanvas().getScene().getDrawableSymmetryPlane(index).setColor(
                (index == 0) ? DrawableFace.SKIN_COLOR_PRIMARY.darker() : DrawableFace.SKIN_COLOR_SECONDARY.darker()
        );
    }
    
    protected void updatePrecision(HumanFace face) {
        if (face == null) {
            return;
        }
        
        MeshModel clone = new MeshModel(face.getMeshModel());
        
        for (MeshFacet facet: clone.getFacets()) { // invert mesh
            facet.getVertices().parallelStream().forEach(v -> {
                v.getPosition().set(face.getSymmetryPlane().reflectOverPlane(v.getPosition()));
            });
        }
        
        face.computeKdTree(false);
        HausdorffDistance visitor = new HausdorffDistance(
                face.getKdTree(), 
                Strategy.POINT_TO_POINT_DISTANCE_ONLY, 
                false, // relative
                true, // parallel
                false // crop
        );
        clone.compute(visitor);
        controlPanel.updateHausdorffDistanceStats(visitor.getStats(), getCanvas().getHumanFace(0) == face);
    }

    private int drawPointSamples(int faceSlot, int cloudSlot, int numSamples) {
        HumanFace face = getCanvas().getHumanFace(faceSlot);
        if (face == null) {
            return -1;
        }                
        
        PointSampling sampling = getSamplingStrategy(numSamples, face.getCurvature());
        face.getMeshModel().compute(sampling);
        
        if (sampling.getClass() == NoSampling.class) { // don't show
            if (cloudSlot != -1) {
                getScene().setOtherDrawable(cloudSlot, null);
            }
            return -1;
        } else {
            if (cloudSlot == -1) {
                cloudSlot = getCanvas().getScene().getFreeSlot();
            }
            getScene().setOtherDrawable(cloudSlot, new DrawablePointCloud(sampling.getSamples()));
            return cloudSlot;
        }
    }
    
    private PointSampling getSamplingStrategy(int numSamples, Curvature samplingCurvature) {
        String st = controlPanel.getPointSamplingStrategy();
        
        if (st.equals(SymmetryPanel.POINT_SAMPLING_STRATEGIES[0])) {
            return new RandomSampling(numSamples);
        } else if (st.equals(SymmetryPanel.POINT_SAMPLING_STRATEGIES[1])) {
            return new CurvatureSampling(samplingCurvature, CurvatureSampling.CurvatureAlg.MEAN, numSamples);
        } else if (st.equals(SymmetryPanel.POINT_SAMPLING_STRATEGIES[2])) {
            return new CurvatureSampling(samplingCurvature, CurvatureSampling.CurvatureAlg.GAUSSIAN, numSamples);
        } else if (st.equals(SymmetryPanel.POINT_SAMPLING_STRATEGIES[3])) {
            return new CurvatureSampling(samplingCurvature, CurvatureSampling.CurvatureAlg.MAX, numSamples);
        } else if (st.equals(SymmetryPanel.POINT_SAMPLING_STRATEGIES[4])) {
            return new CurvatureSampling(samplingCurvature, CurvatureSampling.CurvatureAlg.MIN, numSamples);
        } else {
            return null;
        }
    }
    
}
