package cz.fidentis.analyst.visitors.mesh.sampling;

import cz.fidentis.analyst.mesh.core.MeshFacet;
import cz.fidentis.analyst.mesh.core.MeshPoint;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

/**
 * Random sampling when vertices are selected randomly.
 * 
 * @author Radek Oslejsek
 */
public class RandomSampling extends PointSampling {
    
    private List<MeshPoint> allVertices = new ArrayList<>();

    /**
     * Constructor for PERCENTAGE point sampling type.
     * 
     * @param perc Percentage - a value in (0.0, 1.0&gt;
     * @throws IllegalArgumentException if the input parameter is wrong
     */
    public RandomSampling(double perc) {
        super(perc);
    }
    
    /**
     * Constructor for MAX_VERTICES point sampling type.
     * 
     * @param max Maximal number of vertices. Must be bigger than zero
     * @throws IllegalArgumentException if the input parameter is wrong
     */
    public RandomSampling(int max) {
        super(max);
    }
     
    @Override
    public void visitMeshFacet(MeshFacet facet) {
        if (facet != null) {
            allVertices.addAll(facet.getVertices());
        }
    }
    
    @Override
    public List<MeshPoint> getSamples() {
        int numReducedVertices = getNumDownsampledPoints(allVertices.size());
        
        if (allVertices.size() == numReducedVertices) {
            return Collections.unmodifiableList(allVertices);
        }
        
        // generate randomly ordered indexes:
        List<Integer> range = IntStream.range(0, allVertices.size()).boxed().collect(Collectors.toCollection(ArrayList::new));
        Collections.shuffle(range);
        
        if (numReducedVertices < allVertices.size() / 2) { // copy indices
            MeshPoint[] array = new MeshPoint[allVertices.size()];
            range.stream().limit(numReducedVertices).forEach(
                    i -> array[i] = allVertices.get(i)
            );
            return Arrays.stream(array).filter(p -> p != null).collect(Collectors.<MeshPoint>toList());
        } else { // remove indices
            List<MeshPoint> copy = new ArrayList<>(allVertices);
            range.stream().limit(allVertices.size() - numReducedVertices).forEach(
                    i -> copy.set(i, null)
            );
            return copy.parallelStream().filter(p -> p != null).collect(Collectors.<MeshPoint>toList());
        }        
    }
    
    @Override
    public String toString() {
        return "random " + super.toString();
    }
    
}
