package cz.fidentis.analyst.visitors.mesh.sampling;

import cz.fidentis.analyst.mesh.MeshVisitor;
import cz.fidentis.analyst.mesh.core.MeshFacet;
import cz.fidentis.analyst.mesh.core.MeshPoint;
import java.util.List;

/**
 * Point sampling strategies used to reduce the size of meshes.
 * 
 * @author Radek Oslejsek
 */
public abstract class PointSampling extends MeshVisitor {

    /**
     * If point sampling is used, then this configuration parameter 
     * encodes whether the reduction of mesh size is expressed as the percentage 
     * of the original size (number of vertices) or as the maximal number of vertices
     * to be used.
     * 
     * @author Radek Oslejsek
     */
    public enum PointSamplingType {
        PERCENTAGE,
        MAX_VERTICES
    };
    
    private final PointSamplingType samplingType;
    private final double samplingLimit;
    
    /** 
     * Constructor for no point sampling.
     */
    public PointSampling() {
        this.samplingType = null;
        this.samplingLimit = 0.0;
    }
    
    /**
     * Constructor for PERCENTAGE point sampling type.
     * 
     * @param perc Percentage - a value in (0.0, 1.0&gt;
     * @throws IllegalArgumentException if the input parameter is wrong
     */
    public PointSampling(double perc) {
        if (perc <= 0.0 || perc > 1) {
            throw new IllegalArgumentException("perc");
        }
        this.samplingType = PointSamplingType.PERCENTAGE;
        this.samplingLimit = perc;
    }
    
    /**
     * Constructor for MAX_VERTICES point sampling type.
     * 
     * @param max Maximal number of vertices. Must be bigger than zero
     * @throws IllegalArgumentException if the input parameter is wrong
     */
    public PointSampling(int max) {
        if (max <= 0) {
            throw new IllegalArgumentException("max");
        }
        this.samplingType = PointSamplingType.MAX_VERTICES;
        this.samplingLimit = max;
    }
    
    @Override
    public abstract void visitMeshFacet(MeshFacet facet);    
    
    /**
     * Returns a list of vertices reduced according to the strategy. 
     * The returned mesh points are backed by original points.
     * 
     * @return selected vertices of inspected meshes
     */
    public abstract List<MeshPoint> getSamples();
    
    @Override
    public String toString() {
        if (this.samplingType == PointSamplingType.PERCENTAGE) {
            return "sampling to " + (int) (samplingLimit * 100) + "%";
        } else {
            return "sampling of " + (int) samplingLimit + " points";
        }
    }
    
    /**
     * Returns number of points to be returned after downsampling.
     * 
     * @param origPoints Original number of vertices
     * @return number of points to be returned after downsampling.
     */
    protected int getNumDownsampledPoints(int origPoints) {
        switch (this.samplingType) {
            case PERCENTAGE:
                return (int) (origPoints * this.samplingLimit);
            case MAX_VERTICES:
                //nt limit = (int) this.undersamplingLimit;
                //return (limit <= origVertices) ? limit : origVertices;
                return Math.min((int) this.samplingLimit, origPoints);
            default:
                return 0;
        }
    }
    
}
