package cz.fidentis.analyst.visitors.mesh.sampling;

import cz.fidentis.analyst.mesh.core.MeshFacet;
import cz.fidentis.analyst.mesh.core.MeshPoint;
import cz.fidentis.analyst.visitors.mesh.Curvature;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.PriorityQueue;
import java.util.stream.Collectors;

/**
 * A relevance-based point sampling using highest curvature.
 * {@see https://graphics.stanford.edu/papers/zipper/zipper.pdf}
 * 
 * <p>
 * This visitor <b>is not thread-safe</b> for the efficiency reasons, i.e., 
 * a single instance of the visitor cannot be used to inspect multiple meshes 
 * simultaneously (sequential inspection is okay).
 * </p>
 * 
 * @author Radek Oslejsek
 * @author Natalia Bebjakova
 */
public class CurvatureSampling extends PointSampling {
    
    /**
     * Curvature algorithm used for the selection of the top X significant points.
     * 
     * @author Radek Oslejsek
     */
    public enum CurvatureAlg {
        MEAN,
        GAUSSIAN,
        MAX,
        MIN
    }

    //private final int maxPoints;
    
    private final Curvature curvatureVisitor;
    
    private final CurvatureAlg curvatureAlg;
    
    private List<VertCurvature> significantPoints;
    
    /**
     * Constructor.
     * 
     * @param cAlg Curvature strategy to be used for the selection of significant points.
     * @param max Maximal number of significant points
     * @throws IllegalArgumentException if the {@code curbatureVisitor} is missing
     */
    public CurvatureSampling(CurvatureAlg cAlg, int max) {
        this(new Curvature(), cAlg, max);
    }
    
    /**
     * Constructor.
     * 
     * @param cAlg Curvature strategy to be used for the selection of significant points.
     * @param perc Maximal number of significant points as percents of the original size
     */
    public CurvatureSampling(CurvatureAlg cAlg, double perc) {
        this(new Curvature(), cAlg, perc);
    }
    
    /**
     * Constructor.
     * 
     * @param curvatureVisitor Curvature. If the object has
     * pre-filled curvatures of meshes, then they are also used for the computation
     * of significant points. In this way, it is possible to reuse already computed
     * curvatures and skip calling the {@link #visitMeshFacet} inspection method.
     * @param cAlg Curvature strategy to be used for the selection of significant points.
     * @param max Maximal number of significant points
     * @throws IllegalArgumentException if the {@code curbatureVisitor} is missing
     */
    public CurvatureSampling(Curvature curvatureVisitor, CurvatureAlg cAlg, int max) {
        super(max);
        
        if (curvatureVisitor == null) {
            throw new IllegalArgumentException("curvatureVisitor");
        }

        this.curvatureVisitor = curvatureVisitor;
        this.curvatureAlg = cAlg;
    }
    
    /**
     * Constructor.
     * 
     * @param curvatureVisitor Curvature. If the object has
     * pre-filled curvatures of meshes, then they are also used for the computation
     * of significant points. In this way, it is possible to reuse already computed
     * curvatures and skip calling the {@link #visitMeshFacet} inspection method.
     * @param cAlg Curvature strategy to be used for the selection of significant points.
     * @param perc Maximal number of significant points as percents of the original size
     */
    public CurvatureSampling(Curvature curvatureVisitor, CurvatureAlg cAlg, double perc) {
        super(perc);
        
        if (curvatureVisitor == null) {
            throw new IllegalArgumentException("curvatureVisitor");
        }

        this.curvatureVisitor = curvatureVisitor;
        this.curvatureAlg = cAlg;
    }
    
    @Override
    public boolean isThreadSafe() {
        return false;
    }
    
    @Override
    public void visitMeshFacet(MeshFacet facet) {
        significantPoints = null; // clear previous results
        curvatureVisitor.visitMeshFacet(facet); // compute curvature for new inspected facet
    }
    
    @Override
    public List<MeshPoint> getSamples() {
        checkAndComputeSignificantPoints();
        return significantPoints.stream().map(vc -> vc.point).collect(Collectors.toList());
    }
    
    public CurvatureAlg getCurvatureAlg() {
        return this.curvatureAlg;
    }
    
    /**
     * Returns curvatures of selected samples
     * 
     * @return curvatures of selected samples
     */
    public List<Double> getSampledCurvatures() {
        checkAndComputeSignificantPoints();
        return significantPoints.stream().map(vc -> vc.curvature).collect(Collectors.toList());
    }
    
    @Override
    public String toString() {
        return this.curvatureAlg + " curvature " + super.toString();
    }

    protected void checkAndComputeSignificantPoints() {
        if (significantPoints != null) {
            return; // already computed
        }
        
        // fill the priority queue
        final PriorityQueue<VertCurvature> priorityQueue = new PriorityQueue<>();
        
        Map<MeshFacet, List<Double>> curvMap = null;
        switch (curvatureAlg) {
            case MEAN: 
                curvMap = curvatureVisitor.getMeanCurvatures();
                break;
            case GAUSSIAN: 
                curvMap = curvatureVisitor.getGaussianCurvatures();
                break;
            case MAX: 
                curvMap = curvatureVisitor.getMaxPrincipalCurvatures();
                break;
            case MIN: 
                curvMap = curvatureVisitor.getMinPrincipalCurvatures();
                break;
            default:
                throw new IllegalArgumentException("curvatureAlg");
        }

        for (MeshFacet facet: curvMap.keySet()) {
            List<Double> curvs = curvMap.get(facet);
            for (int i = 0; i < curvs.size(); i++) { 
                // store helper objects, replace NaN with 0.0:
                double c = curvs.get(i);
                priorityQueue.add(new VertCurvature(facet.getVertex(i), (Double.isNaN(c) ? 0.0 : c)));
            }  
        }
        
        // select top significant points
        int maxPoints = getNumDownsampledPoints(priorityQueue.size());
        significantPoints = new ArrayList<>(maxPoints);
        for (int i = 0; i < maxPoints; i++) {
            VertCurvature vc = priorityQueue.poll();
            if (vc == null) {
                break; // no more points available
            } else {
                significantPoints.add(vc);
            }
        }
    }
    
    /**
     * Helper class for sorting points with respect to their curvature.
     * @author Radek Oslejsek
     */
    private class VertCurvature implements Comparable<VertCurvature> {
        
        private final double curvature;
        private final MeshPoint point;
        
        VertCurvature(MeshPoint point, double curvature) {
            this.curvature = curvature;
            this.point = point;
        }

        /**
         * Compares this object with the specified object for order. 
         * Curvature is taken into consideration as the primary value (descendant ordering).
         * If the curvature equals, then the objects are sorted randomly (1 is always returned).
         * This approach preserves all points in sorted collections.
         * 
         * @param arg the object to be compared.
         * @return a negative integer, zero, or a positive integer as this object 
         * is less than, equal to, or greater than the specified object.
         */
        @Override
        public int compareTo(VertCurvature arg) {
            int comp = Double.compare(arg.curvature, this.curvature);
            return (comp == 0) ? 1 : comp;
        }
        
        @Override
        public String toString() {
            return ""+curvature;
        }
    }
    
}
