package cz.fidentis.analyst.symmetry;

import cz.fidentis.analyst.mesh.core.MeshPoint;
import java.util.ArrayList;
import java.util.List;
import javax.vecmath.Vector3d;

/**
 * Precomputed values for the symmetry plane estimation.
 * 
 * @author Radek Oslejsek
 */
public class SymmetryCache {
    
    private List<List<Vector3d>> normCosVecCache;
    private List<List<Vector3d>> avgPosCache;
    private List<List<Double>> curRatioCache;
    
    /**
     * Constructor.
     * 
     * @param vertices Mesh vertices 
     * @param curvatures Curvatures in the vertices
     */
    public SymmetryCache(List<MeshPoint> vertices, List<Double> curvatures) {
        if (vertices == null || vertices.isEmpty()) {
            throw new IllegalArgumentException("points");
        }
        
        normCosVecCache = new ArrayList<>(vertices.size());
        avgPosCache = new ArrayList<>(vertices.size());
        curRatioCache = (curvatures == null) ? null : new ArrayList<>(vertices.size());
        
        for (int i = 0; i < vertices.size(); i++) {
            List<Vector3d> cosArray = new ArrayList<>(vertices.size());
            normCosVecCache.add(cosArray);
            
            List<Vector3d> posArray = new ArrayList<>(vertices.size());
            avgPosCache.add(posArray);
            
            List<Double> curArray = null;
            if (curvatures != null) {
                curArray = new ArrayList<>(vertices.size());
                curRatioCache.add(curArray);
            }
            
            for (int j = 0; j < vertices.size(); j++) {
                MeshPoint meshPointI = vertices.get(i);
                MeshPoint meshPointJ = vertices.get(j);
                
                Vector3d ni = new Vector3d(meshPointI.getNormal());
                Vector3d nj = new Vector3d(meshPointJ.getNormal());
                ni.normalize();
                nj.normalize();
                ni.sub(nj);
                ni.normalize();
                cosArray.add(ni);
                
                Vector3d avrg = new Vector3d(meshPointI.getPosition());
                Vector3d aux = new Vector3d(meshPointJ.getPosition());
                avrg.add(aux);
                avrg.scale(0.5);
                posArray.add(avrg);
                if (curvatures != null) {
                    curArray.add(curvatures.get(i) / curvatures.get(j));
                }
            }
        }
    }
    
    /**
     * Returns cached normal vector for cos.
     * @param i index of the i-th significant curvature point
     * @param j index of the j-th significant curvature point
     * @return cached value or null
     */
    public Vector3d getNormCosVec(int i, int j) {
        if (normCosVecCache == null || i >= normCosVecCache.size() || i >= normCosVecCache.size() || i < 0 || j < 0) {
            return null;
        }
        return normCosVecCache.get(i).get(j);
    }
    
    /**
     * Returns cached vector for the computation average normal.
     * @param i index of the i-th significant curvature point
     * @param j index of the j-th significant curvature point
     * @return cached value or null
     */
    public Vector3d getAvgPos(int i, int j) {
        if (avgPosCache == null || i >= avgPosCache.size() || i >= avgPosCache.size() || i < 0 || j < 0) {
            return null;
        }
        return avgPosCache.get(i).get(j);
    }
    
    /**
     * Returns cached curvature ratio.
     * @param i index of the i-th significant curvature point
     * @param j index of the j-th significant curvature point
     * @return cached value or {@code Double.NaN}
     */
    public double getCurRatio(int i, int j) {
        if (curRatioCache == null || i >= curRatioCache.size() || i >= curRatioCache.size() || i < 0 || j < 0) {
            return Double.NaN;
        }
        return curRatioCache.get(i).get(j);
    }
    
}
