package cz.fidentis.analyst.symmetry;

import cz.fidentis.analyst.mesh.core.MeshPoint;
import java.util.List;
import javax.vecmath.Vector3d;

/**
 * Symmetry plane with votes used for the decision about symmetry estimate of 3D models.
 *
 * @author Natalia Bebjakova
 * 
 */
public class ApproxSymmetryPlane extends Plane implements Comparable<ApproxSymmetryPlane> {
    
    private int votes;

    /**
     * Constructor.
     * 
     * @param vertices Mesh vertices 
     * @param cache Precomputed values form mesh vertices
     * @param config Symmetry plane configuration
     * @param i index of the first significant point used for the plane construction
     * @param j index of the second significant point used for the plane construction
     * @param maxDist Distance limit
     * @throws UnsupportedOperationException if the symmetry plane cannot be created
     */
    public ApproxSymmetryPlane(List<MeshPoint> vertices, SymmetryCache cache, SymmetryConfig config, int i, int j, double maxDist) throws UnsupportedOperationException {
        if (i == j) {
            throw new UnsupportedOperationException();
        }
        setNormAndDist(vertices, cache, config, i, j);
        computeVotes(vertices, cache, config, maxDist);
    }

    /**
     * returns number of votes that were given to plane while computing the symmetry 
     * 
     * @return Number of votes 
     */
    public int getVotes() {
        return votes;
    }
    
    private void setNormAndDist(List<MeshPoint> vertices, SymmetryCache cache, SymmetryConfig config, int i, int j) {
        MeshPoint meshPointI = vertices.get(i);
        MeshPoint meshPointJ = vertices.get(j);
                
        // accpet only point pairs with significantly different curvatures (WTF?)
        double minRatio = config.getMinCurvRatio();
        double maxRatio = 1.0 / minRatio;
        double ratioIJ = cache.getCurRatio(i, j);
        if (Double.isFinite(ratioIJ) && (ratioIJ < minRatio || ratioIJ > maxRatio)) {
            throw new UnsupportedOperationException();
        }
                
        Vector3d p1 = new Vector3d(meshPointI.getPosition());
        Vector3d p2 = new Vector3d(meshPointJ.getPosition());
        Vector3d normal = new Vector3d(p1);
        normal.sub(p2);
        normal.normalize(); 
        
        // accpect only point pair with oposite normals along with the plane normal:
        double normCos = cache.getNormCosVec(i, j).dot(normal);
        if (Math.abs(normCos) < config.getMinNormAngleCos()) {
            throw new UnsupportedOperationException();
        }
        
        setDistance(-normal.dot(cache.getAvgPos(i, j))); 
        setNormal(normal);
    }

    /**
     * Computes votes for given plane 
     *
     * @param sigPoints Mesh vertices with the most significant curvature
     * @param config Symmetry plane configuration
     * @param maxDist Distance limit
     */
    private void computeVotes(List<MeshPoint> vertices, SymmetryCache cache, SymmetryConfig config, double maxDist) {
        normalize();
        Vector3d normal = getNormal();
        double d = getDistance();
        double maxCurvRatio = 1.0 / config.getMinCurvRatio();
        
        for (int i = 0; i < vertices.size(); i++) {
            for (int j = 0; j < vertices.size(); j++) {
                if (i == j) {
                    continue;
                }
                
                double ratioIJ = cache.getCurRatio(i, j);
                if (Double.isFinite(ratioIJ) && (ratioIJ < config.getMinCurvRatio() || ratioIJ > maxCurvRatio)) {
                    continue;
                }
                
                double normCos = cache.getNormCosVec(i, j).dot(normal);
                if (Math.abs(normCos) < config.getMinNormAngleCos()) {
                    continue;
                }
                
                // Caching this part doesn't improve efficiency anymore:
                Vector3d vec = new Vector3d(vertices.get(i).getPosition());
                vec.sub(vertices.get(j).getPosition());
                vec.normalize();
                double cos = vec.dot(normal);
                if (Math.abs(cos) < config.getMinAngleCos()) {
                    continue;
                }
                
                Vector3d avg = cache.getAvgPos(i, j);
                double dist = Math.abs(normal.dot(avg) + d);
                if (dist <= maxDist) {                 
                    votes++;  
                }   
            }
        }
    }
    
    /**
     * Enables to compare two approximate planes due to number of votes 
     * 
     * @param other plane to be compared 
     * @return number that decides which plane has more votes 
     */
    @Override
    public int compareTo(ApproxSymmetryPlane other) {
        return Integer.compare(votes, other.votes);
    }
    
    @Override
    public String toString() {
        return this.getNormal() + " " + getDistance() + " " + votes;
    }
}
