package cz.fidentis.analyst;

import java.io.File;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * This class encapsulates data important for project (re)storing such as paths
 * to faces or which tabs were opened
 * 
 * @author Matej Kovar
 */
public class ProjectConfiguration {
    
    /* Paths to loaded models */
    private List<Path> paths = new ArrayList<>();
    
    private List<String> singleTabFaces = new ArrayList<>();
    
    // f.e. [face1 : [face2, face3], face2 : [face4, face5, face6], face3 : [face4]...]
    private Map<String, List<String>> faceToFaceTabFaces = new HashMap<>();
    
    //private Map<HumanFace, SingleFaceTab> singleFaceTabs = new HashMap<>();
    //private Map<HumanFace, FaceToFaceTab> faceToFaceTabs = new HashMap<>();
    

    public List<Path> getPaths() {
        return paths;
    }

    public void setPaths(List<Path> paths) {
        this.paths = paths;
    }
       
    /**
     * Adds path to paths
     * @param path Path to be added
     * @return true if path was successfully added 
     */
    public boolean addPath(Path path) {
        if (paths.contains(path)) {
            return false;
        }
        paths.add(path);
        return true;
    }
    
    /**
     * Removes specific path from paths
     * @param name String of face
     */
    public void removePath(String name) {
        paths.removeIf(p -> p.toString().substring(
                p.toString().lastIndexOf(File.separatorChar) + 1, 
                p.toString().lastIndexOf('.')
        ).equals(name));
    }
    
    /**
     * Returns path to face with specified name of file
     * @param name String name of file
     * @return Path to face
     */
    public Path getPathToFaceByName(String name) {
        for (Path p : paths) {
            if (p.toString().substring(p.toString().lastIndexOf(File.separatorChar) + 1, p.toString().lastIndexOf('.')).equals(name)) {
                return p;
            }
        }
        return null;
    }

    /**
     * Opens all files in paths
     * @return List<File> list of files with faces
     */
    public List<File> openFiles() {
        List<File> f = new ArrayList<>();
            
        paths.forEach(p -> {
            if (p.toFile().exists()) {
                f.add(p.toFile());
            }
        });
        
        return f;
    }
    
    /**
     * Removes all paths
     */
    public void clearPaths() {
        paths.clear();
    }

    public List<String> getSingleTabFaces() {
        return singleTabFaces;
    }

    public void setSingleTabFaces(List<String> singleTabFaces) {
        this.singleTabFaces = singleTabFaces;
    }

    public Map<String, List<String>> getFaceToFaceTabFaces() {
        return faceToFaceTabFaces;
    }

    public void setFaceToFaceTabFaces(Map<String, List<String>> faceToFaceTabFaces) {
        this.faceToFaceTabFaces = faceToFaceTabFaces;
    }

    /**
     * Adds SingleFace tab
     * @param name String name of face
     */
    public void addSingleFaceTabFace(String name) {
        singleTabFaces.add(name);
    }
    
    /**
     * Adds FaceToFace tab
     * @param name1 String name of first face
     * @param name2 String name of second face
     */
    public void addFaceToFaceTabFace(String name1, String name2) {
        if (faceToFaceTabFaces.containsKey(name1)) {
            faceToFaceTabFaces.get(name1).add(name2);
        } else {
            List<String> faces = new ArrayList<>();
            faces.add(name2);
            faceToFaceTabFaces.put(name1, faces);
        }
    }
    
    /**
     * Removes SingleFace tab
     * @param name String name of face
     */
    public void removeFaceTab(String name) {
        singleTabFaces.remove(name);
    }
    
    /**
     * Removes FaceToFace tab
     * @param name1 String name of first face
     * @param name2 String name of second face
     */
    public void removeFaceToFaceTab(String name1, String name2) {
        faceToFaceTabFaces.get(name1).remove(name2);
        if (faceToFaceTabFaces.get(name1).isEmpty()) {
            faceToFaceTabFaces.remove(name1);
        }
    }
}
