package cz.fidentis.analyst.mesh.core;

import java.util.Objects;
import javax.vecmath.Vector3d;

/**
 * MeshPoint represents a point with position, normal, and texture coordinates.
 *
 * @author Matej Lukes
 */
public class MeshPointImpl implements MeshPoint {
    
    private Vector3d position, normal, texCoord;

    /**
     * constructor of meshPoint
     *
     * @param position position of MeshPoint
     * @param normal   normal of MeshPoint
     * @param texCoord coordinates in texture
     */
    public MeshPointImpl(Vector3d position, Vector3d normal, Vector3d texCoord) {
        if (position == null) {
            throw new IllegalArgumentException("position cannot be null");
        } else {
            this.position = new Vector3d(position);
        }
        
        if (normal != null) {
            this.normal = new Vector3d(normal);
        }
        
        if (texCoord != null) {
            this.texCoord = new Vector3d(texCoord);
        }        
    }

    /**
     * copy constructor of meshPoint
     * 
     * @param meshPoint copied meshPoint
     */
    public MeshPointImpl(MeshPoint meshPoint) {
        this(meshPoint.getPosition(), meshPoint.getNormal(), meshPoint.getTexCoord());
    }

    @Override
    public Vector3d getNormal() {
        return normal;
    }

    @Override
    public Vector3d getPosition() {
        return position;
    }

    @Override
    public Vector3d getTexCoord() {
        return texCoord;
    }
    
    @Override
    public double distanceTo(MeshPoint point) {
        return MeshPoint.distance(position, point.getPosition());
    }

    @Override
    public MeshPoint subtractPosition(MeshPoint subtrahend) {
        return subtractPosition(subtrahend.getPosition());
    }

    @Override
    public MeshPoint subtractPosition(Vector3d subtrahend) {
        Vector3d newPosition = new Vector3d(position);
        newPosition.sub(subtrahend);
        return new MeshPointImpl(newPosition, normal, texCoord);
    }

    @Override
    public MeshPoint addPosition(MeshPoint addend) {
        return addPosition(addend.getPosition());
    }

    @Override
    public MeshPoint addPosition(Vector3d addend) {
        Vector3d newPosition = new Vector3d(position);
        newPosition.add(addend);
        return new MeshPointImpl(newPosition, normal, texCoord);
    }

    @Override
    public MeshPoint multiplyPosition(double number) {
        return new MeshPointImpl(new Vector3d(this.getPosition().x * number,
                this.getPosition().y * number, this.getPosition().z * number),
                normal, texCoord);
    }

    @Override
    public MeshPoint dividePosition(double number) {
        return new MeshPointImpl(new Vector3d(this.getPosition().x / number, this.getPosition().y / number,
                this.getPosition().z / number), normal, texCoord);
    }
    
    @Override
    public MeshPoint crossProduct(MeshPoint meshPoint) {
        return new MeshPointImpl(
                new Vector3d(this.position.y * meshPoint.getPosition().z - this.position.z * meshPoint.getPosition().y,
                        this.position.z * meshPoint.getPosition().x - this.position.x * meshPoint.getPosition().z,
                        this.position.x * meshPoint.getPosition().y - this.position.y * meshPoint.getPosition().x),
                normal, texCoord);
    }

    @Override
    public double dotProduct(MeshPoint meshPoint) {
        return (this.position.x * meshPoint.getPosition().x + this.position.y * meshPoint.getPosition().y + this.position.z * meshPoint.getPosition().z);
    }
    
    @Override
    public double abs() {
        return Math.sqrt(this.getPosition().x * this.getPosition().x + 
                this.getPosition().y * this.getPosition().y + this.getPosition().z * this.getPosition().z);
    }

    @Override
    public MeshPoint subtractNormal(MeshPoint subtrahend) {
        return subtractNormal(subtrahend.getNormal());
    }

    @Override
    public MeshPoint subtractNormal(Vector3d subtrahend) {
        Vector3d newNormal = new Vector3d(normal);
        newNormal.sub(subtrahend);
        newNormal.normalize();
        return new MeshPointImpl(position, newNormal, texCoord);
    }

    @Override
    public MeshPoint addNormal(MeshPoint addend) {
        return addNormal(addend.getNormal());
    }

    @Override
    public MeshPoint addNormal(Vector3d addend) {
        Vector3d newNormal = new Vector3d(normal);
        newNormal.add(addend);
        newNormal.normalize();
        return new MeshPointImpl(position, newNormal, texCoord);
    }

    @Override
    public boolean equals(Object obj) {
        if (!(obj instanceof MeshPointImpl)) {
            return false;
        }

        MeshPointImpl other = (MeshPointImpl) obj;
        
        if (!this.position.equals(other.position)) {
            return false;
        }
        
        if (this.normal == null) {
            if (other.normal != null) {
                return false;
            } 
        } else {
            if (other.normal == null) {
                return false;
            } 
            if (!this.normal.equals(other.normal)) {
                return false;
            }
        }
        
        if (this.texCoord == null) {
            if (other.texCoord != null) {
                return false;
            } 
        } else {
            if (other.texCoord == null) {
                return false;
            } 
            if (!this.texCoord.equals(other.texCoord)) {
                return false;
            }
        }
        
        return true;
    }

    @Override
    public int hashCode() {
        int hash = 5;
        hash = 67 * hash + Objects.hashCode(this.position);
        hash = 67 * hash + Objects.hashCode(this.normal);
        hash = 67 * hash + Objects.hashCode(this.texCoord);
        return hash;
    }

    @Override
    public String toString() {
        StringBuilder ret = new StringBuilder();
        ret.append("p=(").append(String.format("%.2f", position.x)).
                append("  ").append(String.format("%.2f", position.y)).
                append("  ").append(String.format("%.2f", position.z)).
                append(") ");
        if (normal != null) {
            ret.append("n=(").append(String.format("%.2f", normal.x)).
                    append("  ").append(String.format("%.2f", normal.y)).
                    append("  ").append(String.format("%.2f", normal.z)).
                    append(") ");
        }
        if (texCoord != null) {
            ret.append("t=(").append(String.format("%.2f", texCoord.x)).
                    append("  ").append(String.format("%.2f", texCoord.y)).
                    append("  ").append(String.format("%.2f", texCoord.z)).
                    append(") ");
        }
        return ret.toString();
    }
}
