package cz.fidentis.analyst.symmetry;

import cz.fidentis.analyst.core.ControlPanel;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.Arrays;
import java.util.DoubleSummaryStatistics;
import javax.swing.ImageIcon;
import javax.swing.JOptionPane;
import static javax.swing.JOptionPane.INFORMATION_MESSAGE;

/**
 * Control panel for symmetry plane.
 *
 * @author Natalia Bebjakova
 * @author Radek Oslejsek
 */
public class SymmetryPanel extends ControlPanel  {
    
    /*
     * Handled actions
     */
    public static final String ACTION_COMMAND_RECOMPUTE_FROM_MESH = "Mesh vertices";
    public static final String ACTION_COMMAND_COMPUTE_FROM_FPS = "Feature points";
    public static final String ACTION_COMMAND_POINT_SAMPLING_STRENGTH = "Point sampling strength";
    public static final String ACTION_COMMAND_POINT_SAMPLING_STRATEGY = "Point sampling strategy";
    
    public static final String[] POINT_SAMPLING_STRATEGIES = new String[] {
        "Uniform Space Sampling",
        "Random Sampling",
        "Mean Curvature",
        "Gaussian Curvature",
        "Max Curvature",
        "Min Curvature"
    };
    
    public static final String[] ALGORITHM = new String[] {
        "Fast for manifold mesh",
        "Robust for point clouds",
        "Robust for manifold mesh",
    };
    
    /*
     * Mandatory design elements
     */
    public static final String ICON = "symmetry28x28.png";
    public static final String NAME = "Symmetry";
    
    
    /**
     * Creates new form SymmetryPanelNew
     */
    public SymmetryPanel(ActionListener action) {
        this.setName(NAME);
        initComponents();
        
        Arrays.stream(POINT_SAMPLING_STRATEGIES).forEach(v -> jComboBox2.addItem(v));
        jComboBox2.setSelectedIndex(0);
        jComboBox2.addActionListener(createListener(action, ACTION_COMMAND_POINT_SAMPLING_STRATEGY));
        
        Arrays.stream(ALGORITHM).forEach(v -> jComboBox3.addItem(v));
        jComboBox3.setSelectedIndex(0);
        
        jButton2.addActionListener((ActionEvent e) -> { 
            action.actionPerformed(new ActionEvent( // recompute
                    e.getSource(), 
                    ActionEvent.ACTION_PERFORMED,
                    selectableComboBox1.getSelectedItem().toString()
            )); 
        });
        
        jButtonInfo1.addActionListener((ActionEvent e) -> { 
            this.showSignPoints1Help();
        });
        
        jButtonInfo2.addActionListener((ActionEvent e) -> { 
            this.showSignPoints2Help();
        });
        
        jButtonInfo3.addActionListener((ActionEvent e) -> { 
            this.showAlgorithmHelp();
        });
        
        selectableComboBox1.addItem(SymmetryPanel.ACTION_COMMAND_RECOMPUTE_FROM_MESH, true);
        selectableComboBox1.addItem(SymmetryPanel.ACTION_COMMAND_COMPUTE_FROM_FPS, true);
        selectableComboBox1.setSelectedIndex(0);
        
        jComboBox3.addActionListener((ActionEvent e) -> {  // set default values
            setDefaultValues(action);
        });
        
        setDefaultValues(action);
    }
    
    /**
     * aaa
     * @return aaa
     */
    public String getAlgorithm() {
        return ALGORITHM[jComboBox3.getSelectedIndex()];
    }
    
    @Override
    public ImageIcon getIcon() {
        return getStaticIcon();
    }
    
    /**
     * Return the number of point samples for the fist phase
     * @return the number of point samples
     */
    public int getPointSamplingStrength1() {
        return (Integer) spinSlider1.getValue();
    }
    
    /**
     * Return the number of point samples for the second phase
     * @return the number of point samples
     */
    public int getPointSamplingStrength2() {
        return (Integer) spinSlider2.getValue();
    }
    
    /**
     * Return selected point sampling strategy
     * @return selected point sampling strategy
     */
    public String getPointSamplingStrategy() {
        return POINT_SAMPLING_STRATEGIES[jComboBox2.getSelectedIndex()];
    }
    
    /**
     * Static implementation of the {@link #getIcon()} method.
     * 
     * @return Control panel icon
     */
    public static ImageIcon getStaticIcon() {
        return new ImageIcon(SymmetryPanel.class.getClassLoader().getResource("/" + ICON));
    }

    /**
     * Enables/disables the selection for the computation of the symmetry from feature points.
     * @param on {@code true} = enable
     */
    public void setComputeFromFPs(boolean on) {
        selectableComboBox1.setItem(SymmetryPanel.ACTION_COMMAND_COMPUTE_FROM_FPS, on);
    }
    
    /**
     * Updates GUI elements that display statistical data about the calculated Hausdorff distance.
     * 
     * @param stats Statistical data of the ordinary Hausdorff distance
     * @param first First face
     */
    public void updateHausdorffDistanceStats(DoubleSummaryStatistics stats, boolean first) {
        if (first) {
            jTextField1.setText(String.format("%.2f", stats.getAverage()));
        } else {
            jTextField2.setText(String.format("%.2f", stats.getAverage()));
        }
    }
    
    private void setDefaultValues(ActionListener action) {
        if (getAlgorithm().equals(ALGORITHM[1])) { //  new robust
            spinSlider1.initInteger(100, 10, 1000, 1);
            spinSlider2.initInteger(1000, 10, 5000, 1);
            spinSlider2.setEnabled(true);
        } else if (getAlgorithm().equals(ALGORITHM[2])) { // new robust with curvature
            spinSlider1.initInteger(200, 10, 500, 1);
            spinSlider2.initInteger(200, 10, 500, 1);
            spinSlider2.setEnabled(true);
        } else if (getAlgorithm().equals(ALGORITHM[0])) { // old fast
            spinSlider1.initInteger(200, 10, 500, 1);
            spinSlider2.initInteger(0, 0, 100, 1);
            spinSlider2.setEnabled(false);
        }
        
        spinSlider1.addSpinnerListener((ActionEvent e) -> {  // show samples in 3D
            action.actionPerformed(new ActionEvent(
                    spinSlider1, 
                    ActionEvent.ACTION_PERFORMED, 
                    ACTION_COMMAND_POINT_SAMPLING_STRENGTH)
            ); 
        });
    }
    
    private void showSignPoints1Help() {
        JOptionPane.showMessageDialog(this,
                "The number of downsampled points" + System.lineSeparator()
                        +"used for the estimation of candidate planes" + System.lineSeparator()
                        + System.lineSeparator()
                        + "Higher number → longer calculation, possibly more accurate result." + System.lineSeparator()
                        + "Lower number → shorter calculation, possibly less accurate result.", 
                "Point sampling - first phase",
                0, 
                new ImageIcon(getClass().getResource("/points.png"))
        );                
    }
     
    private void showSignPoints2Help() {
        JOptionPane.showMessageDialog(this,
                "The number of downsampled points" + System.lineSeparator()
                        + "used for the best candidate selection" + System.lineSeparator()
                        + System.lineSeparator()
                        + "Higher number → longer calculation, possibly more accurate result." + System.lineSeparator()
                        + "Lower number → shorter calculation, possibly less accurate result.", 
                "Point sampling - second phase",
                0, 
                new ImageIcon(getClass().getResource("/points.png"))
        );                
    }
    
    private void showAlgorithmHelp() {
        JOptionPane.showMessageDialog(this,
                "<html>"
                        + "All the algorithms provide similar results.<br/>"
                        + "But the time requirements differ and also<br/>"
                        + "they can fail in some situations.<br/>"
                        + "<br/>"
                        + "<strong>" + ALGORITHM[0] + "</strong>: Fast implemenation<br/>"
                        + "taken from the old FIDENTIS. Uses Gaussian curvature,<br/>"
                        + "orientation of normal vectors, and the location<br/>"
                        + "of points to prune candidate planes and to measure<br/>"
                        + "their quiality. But may fail if the data ara noisy<br/>"
                        + "or incomplete.<br/>"
                        + "<br/>"
                        + "<strong>" + ALGORITHM[1] + "</strong>: Slower but robust<br/>"
                        + "implemenation. Works with point clouds<br/>"
                        + "(manifold mesh is not required).<br/>"
                        + "<br/>"
                        + "<strong>" + ALGORITHM[2] + "</strong>: The slowest implemenation.<br/>"
                        + "It is robust and should deal very well with incomlete data,<br/>"
                        + "e.g., if only a fragment of a face is available.<br/>"
                        + "</html>",
                "Symmetry estimation algorithm",
                INFORMATION_MESSAGE, 
                null
        );                
    }

    /**
     * This method is called from within the constructor to initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is always
     * regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {

        jPanel1 = new javax.swing.JPanel();
        jLabel1 = new javax.swing.JLabel();
        jButtonInfo1 = new javax.swing.JButton();
        jLabel9 = new javax.swing.JLabel();
        jComboBox2 = new javax.swing.JComboBox<>();
        spinSlider1 = new cz.fidentis.analyst.core.SpinSlider();
        jLabel2 = new javax.swing.JLabel();
        jComboBox3 = new javax.swing.JComboBox<>();
        jLabel3 = new javax.swing.JLabel();
        jButtonInfo2 = new javax.swing.JButton();
        spinSlider2 = new cz.fidentis.analyst.core.SpinSlider();
        jButtonInfo3 = new javax.swing.JButton();
        jPanel2 = new javax.swing.JPanel();
        jTextField1 = new javax.swing.JTextField();
        jLabel7 = new javax.swing.JLabel();
        jLabel8 = new javax.swing.JLabel();
        jTextField2 = new javax.swing.JTextField();
        jPanel3 = new javax.swing.JPanel();
        jButton2 = new javax.swing.JButton();
        selectableComboBox1 = new cz.fidentis.analyst.core.combobox.SelectableComboBox();

        jPanel1.setBorder(javax.swing.BorderFactory.createTitledBorder(null, org.openide.util.NbBundle.getMessage(SymmetryPanel.class, "SymmetryPanel.jPanel1.border.title_1"), javax.swing.border.TitledBorder.DEFAULT_JUSTIFICATION, javax.swing.border.TitledBorder.DEFAULT_POSITION, new java.awt.Font("Dialog", 1, 14))); // NOI18N

        org.openide.awt.Mnemonics.setLocalizedText(jLabel1, org.openide.util.NbBundle.getMessage(SymmetryPanel.class, "SymmetryPanel.jLabel1.text_1")); // NOI18N

        jButtonInfo1.setIcon(new javax.swing.ImageIcon(getClass().getResource("/info.png"))); // NOI18N
        org.openide.awt.Mnemonics.setLocalizedText(jButtonInfo1, org.openide.util.NbBundle.getMessage(SymmetryPanel.class, "SymmetryPanel.jButtonInfo1.text")); // NOI18N
        jButtonInfo1.setBorderPainted(false);
        jButtonInfo1.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                jButtonInfo1ActionPerformed(evt);
            }
        });

        org.openide.awt.Mnemonics.setLocalizedText(jLabel9, org.openide.util.NbBundle.getMessage(SymmetryPanel.class, "SymmetryPanel.jLabel9.text")); // NOI18N

        jComboBox2.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                jComboBox2ActionPerformed(evt);
            }
        });

        org.openide.awt.Mnemonics.setLocalizedText(jLabel2, org.openide.util.NbBundle.getMessage(SymmetryPanel.class, "SymmetryPanel.jLabel2.text")); // NOI18N

        org.openide.awt.Mnemonics.setLocalizedText(jLabel3, org.openide.util.NbBundle.getMessage(SymmetryPanel.class, "SymmetryPanel.jLabel3.text")); // NOI18N

        jButtonInfo2.setIcon(new javax.swing.ImageIcon(getClass().getResource("/info.png"))); // NOI18N
        org.openide.awt.Mnemonics.setLocalizedText(jButtonInfo2, org.openide.util.NbBundle.getMessage(SymmetryPanel.class, "SymmetryPanel.jButtonInfo2.text")); // NOI18N
        jButtonInfo2.setBorderPainted(false);

        jButtonInfo3.setIcon(new javax.swing.ImageIcon(getClass().getResource("/info.png"))); // NOI18N
        org.openide.awt.Mnemonics.setLocalizedText(jButtonInfo3, org.openide.util.NbBundle.getMessage(SymmetryPanel.class, "SymmetryPanel.jButtonInfo3.text")); // NOI18N
        jButtonInfo3.setBorderPainted(false);
        jButtonInfo3.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                jButtonInfo3ActionPerformed(evt);
            }
        });

        javax.swing.GroupLayout jPanel1Layout = new javax.swing.GroupLayout(jPanel1);
        jPanel1.setLayout(jPanel1Layout);
        jPanel1Layout.setHorizontalGroup(
            jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(jPanel1Layout.createSequentialGroup()
                .addContainerGap()
                .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addComponent(jLabel1)
                    .addComponent(jLabel3)
                    .addComponent(jLabel2)
                    .addComponent(jLabel9, javax.swing.GroupLayout.PREFERRED_SIZE, 173, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addGroup(javax.swing.GroupLayout.Alignment.TRAILING, jPanel1Layout.createSequentialGroup()
                        .addComponent(jComboBox3, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                        .addComponent(jButtonInfo3))
                    .addGroup(jPanel1Layout.createSequentialGroup()
                        .addComponent(jComboBox2, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                        .addGap(0, 0, Short.MAX_VALUE))
                    .addGroup(jPanel1Layout.createSequentialGroup()
                        .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                            .addComponent(spinSlider1, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                            .addComponent(spinSlider2, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED, 27, Short.MAX_VALUE)
                        .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING, false)
                            .addComponent(jButtonInfo2, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                            .addComponent(jButtonInfo1, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))))
                .addContainerGap())
        );
        jPanel1Layout.setVerticalGroup(
            jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(jPanel1Layout.createSequentialGroup()
                .addContainerGap()
                .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                        .addComponent(jComboBox3, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                        .addComponent(jLabel2))
                    .addComponent(jButtonInfo3))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(jComboBox2, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(jLabel9))
                .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addGroup(jPanel1Layout.createSequentialGroup()
                        .addGap(18, 18, 18)
                        .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING, false)
                            .addComponent(spinSlider1, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                            .addComponent(jButtonInfo1, javax.swing.GroupLayout.PREFERRED_SIZE, 0, Short.MAX_VALUE)))
                    .addGroup(jPanel1Layout.createSequentialGroup()
                        .addGap(21, 21, 21)
                        .addComponent(jLabel1)))
                .addGap(18, 18, 18)
                .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING, false)
                    .addComponent(spinSlider2, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                    .addComponent(jButtonInfo2, javax.swing.GroupLayout.PREFERRED_SIZE, 0, Short.MAX_VALUE)
                    .addGroup(jPanel1Layout.createSequentialGroup()
                        .addGap(8, 8, 8)
                        .addComponent(jLabel3)))
                .addContainerGap(38, Short.MAX_VALUE))
        );

        jPanel2.setBorder(javax.swing.BorderFactory.createTitledBorder(null, org.openide.util.NbBundle.getMessage(SymmetryPanel.class, "SymmetryPanel.jPanel2.border.title"), javax.swing.border.TitledBorder.DEFAULT_JUSTIFICATION, javax.swing.border.TitledBorder.DEFAULT_POSITION, new java.awt.Font("Dialog", 1, 12))); // NOI18N

        jTextField1.setText(org.openide.util.NbBundle.getMessage(SymmetryPanel.class, "SymmetryPanel.jTextField1.text")); // NOI18N

        org.openide.awt.Mnemonics.setLocalizedText(jLabel7, org.openide.util.NbBundle.getMessage(SymmetryPanel.class, "SymmetryPanel.jLabel7.text")); // NOI18N

        org.openide.awt.Mnemonics.setLocalizedText(jLabel8, org.openide.util.NbBundle.getMessage(SymmetryPanel.class, "SymmetryPanel.jLabel8.text")); // NOI18N

        jTextField2.setText(org.openide.util.NbBundle.getMessage(SymmetryPanel.class, "SymmetryPanel.jTextField2.text")); // NOI18N

        javax.swing.GroupLayout jPanel2Layout = new javax.swing.GroupLayout(jPanel2);
        jPanel2.setLayout(jPanel2Layout);
        jPanel2Layout.setHorizontalGroup(
            jPanel2Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(javax.swing.GroupLayout.Alignment.TRAILING, jPanel2Layout.createSequentialGroup()
                .addContainerGap()
                .addComponent(jLabel7)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addComponent(jTextField1, javax.swing.GroupLayout.PREFERRED_SIZE, 55, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addComponent(jLabel8)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addComponent(jTextField2, javax.swing.GroupLayout.PREFERRED_SIZE, 55, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addContainerGap(21, Short.MAX_VALUE))
        );
        jPanel2Layout.setVerticalGroup(
            jPanel2Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(jPanel2Layout.createSequentialGroup()
                .addContainerGap()
                .addGroup(jPanel2Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(jTextField1, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(jLabel7)
                    .addComponent(jLabel8)
                    .addComponent(jTextField2, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addContainerGap(javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
        );

        jPanel3.setBorder(javax.swing.BorderFactory.createTitledBorder(null, org.openide.util.NbBundle.getMessage(SymmetryPanel.class, "SymmetryPanel.jPanel3.border.title"), javax.swing.border.TitledBorder.DEFAULT_JUSTIFICATION, javax.swing.border.TitledBorder.DEFAULT_POSITION, new java.awt.Font("Dialog", 1, 12))); // NOI18N

        jButton2.setFont(new java.awt.Font("Ubuntu", 1, 15)); // NOI18N
        org.openide.awt.Mnemonics.setLocalizedText(jButton2, org.openide.util.NbBundle.getMessage(SymmetryPanel.class, "SymmetryPanel.jButton2.text")); // NOI18N
        jButton2.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                jButton2ActionPerformed(evt);
            }
        });

        javax.swing.GroupLayout jPanel3Layout = new javax.swing.GroupLayout(jPanel3);
        jPanel3.setLayout(jPanel3Layout);
        jPanel3Layout.setHorizontalGroup(
            jPanel3Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(jPanel3Layout.createSequentialGroup()
                .addContainerGap()
                .addComponent(selectableComboBox1, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addComponent(jButton2)
                .addContainerGap(javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
        );
        jPanel3Layout.setVerticalGroup(
            jPanel3Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(jPanel3Layout.createSequentialGroup()
                .addContainerGap()
                .addGroup(jPanel3Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(jButton2)
                    .addComponent(selectableComboBox1, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addContainerGap(javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
        );

        javax.swing.GroupLayout layout = new javax.swing.GroupLayout(this);
        this.setLayout(layout);
        layout.setHorizontalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(layout.createSequentialGroup()
                .addContainerGap()
                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addGroup(layout.createSequentialGroup()
                        .addComponent(jPanel2, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                        .addGap(0, 0, Short.MAX_VALUE))
                    .addGroup(layout.createSequentialGroup()
                        .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.TRAILING)
                            .addComponent(jPanel3, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                            .addComponent(jPanel1, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
                        .addContainerGap())))
        );
        layout.setVerticalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(layout.createSequentialGroup()
                .addContainerGap()
                .addComponent(jPanel3, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addComponent(jPanel1, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addComponent(jPanel2, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addContainerGap(213, Short.MAX_VALUE))
        );
    }// </editor-fold>//GEN-END:initComponents

    private void jButton2ActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_jButton2ActionPerformed
        // TODO add your handling code here:
    }//GEN-LAST:event_jButton2ActionPerformed

    private void jButtonInfo1ActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_jButtonInfo1ActionPerformed
        // TODO add your handling code here:
    }//GEN-LAST:event_jButtonInfo1ActionPerformed

    private void jComboBox2ActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_jComboBox2ActionPerformed
        // TODO add your handling code here:
    }//GEN-LAST:event_jComboBox2ActionPerformed

    private void jButtonInfo3ActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_jButtonInfo3ActionPerformed
        // TODO add your handling code here:
    }//GEN-LAST:event_jButtonInfo3ActionPerformed


    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JButton jButton2;
    private javax.swing.JButton jButtonInfo1;
    private javax.swing.JButton jButtonInfo2;
    private javax.swing.JButton jButtonInfo3;
    private javax.swing.JComboBox<String> jComboBox2;
    private javax.swing.JComboBox<String> jComboBox3;
    private javax.swing.JLabel jLabel1;
    private javax.swing.JLabel jLabel2;
    private javax.swing.JLabel jLabel3;
    private javax.swing.JLabel jLabel7;
    private javax.swing.JLabel jLabel8;
    private javax.swing.JLabel jLabel9;
    private javax.swing.JPanel jPanel1;
    private javax.swing.JPanel jPanel2;
    private javax.swing.JPanel jPanel3;
    private javax.swing.JTextField jTextField1;
    private javax.swing.JTextField jTextField2;
    private cz.fidentis.analyst.core.combobox.SelectableComboBox selectableComboBox1;
    private cz.fidentis.analyst.core.SpinSlider spinSlider1;
    private cz.fidentis.analyst.core.SpinSlider spinSlider2;
    // End of variables declaration//GEN-END:variables
}
