package cz.fidentis.analyst.scene;

import com.jogamp.opengl.GL;
import com.jogamp.opengl.GL2;
import com.jogamp.opengl.glu.GLU;
import com.jogamp.opengl.glu.GLUquadric;
import cz.fidentis.analyst.feature.FeaturePoint;
import java.awt.Color;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Drawable feature points.
 * Points are rendered using smooth filled triangles (see {@link #initRendering(com.jogamp.opengl.GL2)}).
 * 
 * @author Radek Oslejsek
 * @author Daniel Schramm
 * @author Katerina Zarska
 */
public class DrawableFeaturePoints extends Drawable {
    
    public static final Color OFF_THE_MESH_COLOR = Color.GRAY;
    public static final Color CLOSE_TO_MESH_COLOR = Color.ORANGE;
    public static final Color ON_THE_MESH_COLOR = Color.WHITE;
    
    public static final Color FP_DEFAULT_COLOR = Color.LIGHT_GRAY;
    public static final double FP_DEFAULT_SIZE = 3f;
    
    private final double defaultPerimeter;
    
    private static final GLU GLU_CONTEXT = new GLU();
    
    /* feature points */
    private final List<FeaturePoint> featurePoints;
    
    /**
     * feature points with color different from the default color
     */
    private final Map<Integer, Color> specialColors = new HashMap<>();
    
    /**
     * feature points with size different from the default size
     */
    private final Map<Integer, Double> specialSizes = new HashMap<>();
    
    /**
     * feature points rendered differently than the default render mode
     */
    private final Map<Integer, RenderingMode> specialRenderModes = new HashMap<>();
    
    /**
     * Constructor.
     * 
     * @param featurePoints Feature points
     */
    public DrawableFeaturePoints(List<FeaturePoint> featurePoints) {
        this(featurePoints, FP_DEFAULT_COLOR, FP_DEFAULT_SIZE);
    }
    
    /**
     * Constructor.
     * 
     * @param featurePoints Feature points
     * @param defaultColor Default color
     * @param defaultPerimeter Default perimeter
     */
    public DrawableFeaturePoints(List<FeaturePoint> featurePoints, Color defaultColor, double defaultPerimeter) {
        //this.featurePoints = new ArrayList<>(featurePoints);
        this.featurePoints = featurePoints;
        this.defaultPerimeter = defaultPerimeter;
        setColor(defaultColor);
    }
    
    /**
     * Returns color of the feature point.
     * 
     * @param index Index of the feature point
     * @return The color or {@code null}
     */
    public Color getColor(int index) {
        if (index < 0 || index >= featurePoints.size()) {
            return null;
        }
        if (specialColors.containsKey(index)) {
            return specialColors.get(index);
        } else {
            switch (featurePoints.get(index).getRelationToMesh().getPositionType()) {
                case ON_THE_MESH:
                    return ON_THE_MESH_COLOR;
                case CLOSE_TO_MESH:
                    return CLOSE_TO_MESH_COLOR;
                case OFF_THE_MESH:
                    return OFF_THE_MESH_COLOR;
                default:
                    return Color.BLACK;
            }
        }
    }
    
    /**
     * Sets color of the feature point.
     * 
     * @param index Index of the feature point
     * @param color New color of the feature point
     */
    public void setColor(int index, Color color) {
        if (index < 0 || index >= featurePoints.size() || color == null) {
            return;
        }
        if (color.equals(getColor())) {
            specialColors.remove(index);
        } else {
            specialColors.put(index, color);
        }
    }
    
    /**
     * Removes (possible) special color of the feature point.
     * 
     * @param index Index of the feature point
     */
    public void resetColorToDefault(int index) {
        setColor(index, getColor());
    }
    
    /**
     * Removes all individual colors of feature points.
     */
    public void resetAllColorsToDefault() {
        this.specialColors.clear();
    }
    
    /**
     * Returns size of the feature point.
     * 
     * @param index Index of the feature point
     * @return The size or {@code null}
     */
    public Double getSize(int index) {
        if (index < 0 || index >= featurePoints.size()) {
            return null;
        }
        if (specialSizes.containsKey(index)) {
            return specialSizes.get(index);
        } else {
            return this.defaultPerimeter;
        }
    }
    
    /**
     * Sets size of the feature point. 
     * 
     * @param index Index of the feature point
     * @param size New size of the feature point.
     */
    public void setSize(int index, double size) {
        if (index < 0 || index >= featurePoints.size()) {
            return;
        }
        if (size == defaultPerimeter) {
            specialSizes.remove(index);
        } else {
            specialSizes.put(index, size);
        }
    }
    
    /**
     * Removes (possible) special size of the feature point.
     * 
     * @param index Index of the feature point
     */
    public void resetSizeToDefault(int index) {
        setSize(index, defaultPerimeter);
    }
    
    /**
     * Removes all individual sizes of feature points.
     */
    public void resetAllSizesToDefault() {
        specialSizes.clear();
    }
    
    /**
     * Returns render mode of the feature point.
     * 
     * @param index Index of the feature point
     * @return The render mode or {@code null}
     */
    public RenderingMode getRenderMode(int index) {
        if (index < 0 || index >= featurePoints.size()) {
            return null;
        }
        if (specialRenderModes.containsKey(index)) {
            return specialRenderModes.get(index);
        } else {
            return getRenderMode();
        }
    }
    
    /**
     * Sets render mode of the feature point. {@code RenderingMode.NOTHING} hides the point.
     * 
     * @param index Index of the feature point
     * @param renderMode New render mode of the feature point
     */
    public void setRenderMode(int index, RenderingMode renderMode) {
        if (index < 0 || index >= featurePoints.size()) {
            return;
        }
        if (renderMode == getRenderMode()) {
            specialRenderModes.remove(index);
        } else {
            specialRenderModes.put(index, renderMode);
        }
    }
    
    /**
     * Removes (possible) special render mode of the feature point.
     * 
     * @param index Index of the feature point
     */
    public void resetRenderModeToDefault(int index) {
        setRenderMode(index, getRenderMode());
    }
    
    /**
     * Removes all individual render modes of feature points.
     */
    public void resetAllRenderModesToDefault() {
        specialRenderModes.clear();
    }

    @Override
    protected void renderObject(GL2 gl) {
        for (int i = 0; i < featurePoints.size(); i++) {
            FeaturePoint fp = featurePoints.get(i);
            
            Color specialColor = specialColors.get(i);
            Color defaultColor = getColor(i);
            
            float[] rgba = {
                defaultColor.getRed() / 255f, 
                defaultColor.getGreen() / 255f, 
                defaultColor.getBlue() / 255f, 
                defaultColor.getTransparency()};
            
            if (specialColor != null) {
                rgba[0] = specialColor.getRed() / 255f;
                rgba[1] = specialColor.getGreen() / 255f;
                rgba[2] = specialColor.getBlue() / 255f;
                rgba[3] = getTransparency();
            }
            gl.glMaterialfv(GL2.GL_FRONT_AND_BACK, GL2.GL_DIFFUSE, rgba, 0);
            
            RenderingMode specialRenderMode = specialRenderModes.get(i);
            if (specialRenderMode != null && specialRenderMode == RenderingMode.HIDE) { // skip this point
                continue;
            }
            if (specialRenderMode != null) {
                setPolygonMode(gl);
            }
            
            gl.glPushMatrix(); 
            gl.glTranslated(fp.getX(), fp.getY(), fp.getZ());
            GLUquadric center = GLU_CONTEXT.gluNewQuadric();
            GLU_CONTEXT.gluQuadricDrawStyle(center, GLU.GLU_FILL);
            GLU_CONTEXT.gluQuadricNormals(center, GLU.GLU_FLAT);
            GLU_CONTEXT.gluQuadricOrientation(center, GLU.GLU_OUTSIDE);
            GLU_CONTEXT.gluSphere(center, specialSizes.getOrDefault(i, defaultPerimeter), 16, 16);
            GLU_CONTEXT.gluDeleteQuadric(center);
            gl.glPopMatrix();
    
            if (specialRenderMode != null) {
                setPolygonMode(gl); // set default render mode
            }
        }
    }
    
    /**
     * Sets smooth rendering mode with filled triangles mode for feature points
     * @param gl GL context
     */
    /*
    @Override
    protected void initRendering(GL2 gl) {
        setRenderMode(RenderingMode.SMOOTH);
        super.initRendering(gl);        
    }
    */
    
       
    /**
     * @return {@link List} of {@link FeaturePoint}
     */
    public List<FeaturePoint> getFeaturePoints() {
        return featurePoints;
    }
    
    public double getDefaultPerimeter() {
        return defaultPerimeter;
    }
    
    /**
     * Sets feature points
     * @param featurePoints Feature points
     */
    /*
    public void setFeaturePoints(List<FeaturePoint> featurePoints) {
        this.featurePoints = featurePoints;
        List<Color> colors = new ArrayList<>();
        featurePoints.forEach((_item) -> {
            colors.add(getColor());
        });
        this.setFeaturePointsColor(colors);
    } 
    */
    
}
