package cz.fidentis.analyst.scene;

import com.jogamp.opengl.GL;
import com.jogamp.opengl.GL2;
import java.awt.Color;
import java.util.Comparator;

/**
 * A drawable object, i.e., an object with drawing state and capable to 
 * render itself into an OpenGL context.
 * 
 * @author Radek Oslejsek
 * @author Richard Pajersky
 */
public abstract class Drawable {
    
    private boolean display = true;
    
    /* material info */
    private Color color = Color.LIGHT_GRAY;
    private float transparency = 1; // 0 = off, 1 = max
    private Color highlights = new Color(0, 0, 0, 1);
    
    /**
     * Render mode to use, one from {@code GL_FILL}, {@code GL_LINE}, {@code GL_POINT}
     */
    private RenderingMode renderMode = RenderingMode.SMOOTH;
    
    /**
     * Default constructor.
     */
    public Drawable() {
        
    }
    
    /**
     * Copy constructor.
     * @param drawable Original drawable object
     * @throws NullPointException if the input argument is {@code null}
     */
    public Drawable(Drawable drawable) {
        this.color = drawable.color;
        this.highlights = drawable.highlights;
        this.transparency = drawable.transparency;
    }

    /**
     * Comparator for Drawable objects based on transparency. Used for rendering.
     *
     * @author Dominik Racek
     */
    public static class TransparencyComparator implements Comparator<Drawable> {
        @Override
        public int compare(Drawable a, Drawable b) {
            if (a.transparency < b.transparency) {
                return 1;
            }
            if (a.transparency > b.transparency) {
                return -1;
            }
            return 0;
        }
    }

    /**
     * Renders the scene.
     * @param gl OpenGL context
     */
    public void render(GL2 gl) {
        initRendering(gl);
        renderObject(gl);
        finishRendering(gl);
    }
    
    protected void initRendering(GL2 gl) {
        setPolygonMode(gl);
        
        // set color
        float[] rgba = {color.getRed() / 255f, color.getGreen() / 255f, 
            color.getBlue() / 255f , getTransparency()};
        gl.glMaterialfv(GL2.GL_FRONT_AND_BACK, GL2.GL_DIFFUSE, rgba, 0);

    }
    
    protected abstract void renderObject(GL2 gl);

    protected void finishRendering(GL2 gl) {    
    }
        
    /**
     * Shows or hides the drawable.
     * 
     * @param show If {@code true}, then the drawable is shown.
     */
    public void show(boolean show) {
        display = show;
    }
    
    /**
     * 
     * @return {@code true} if the object is included (rendered) in the scene.
     */
    public boolean isShown() {
        return display;
    }
    
    /**
     * Sets color
     * @param color Color
     */
    public void setColor(Color color) {
        this.color = color;
    }

    /**
     * @return {@link Color}
     */
    public Color getColor() {
        return color;
    }
    
    /**
     * @return Current value of transparency
     */
    public float getTransparency() {
        return transparency;
    }

    /**
     * Sets transparency
     * @param transparency Transparency
     */
    public void setTransparency(float transparency) {
        this.transparency = transparency;
    }
    
    /**
     * @return {@link Color} of highlights
     */
    public Color getHighlights() {
        return highlights;
    }

    /**
     * Sets {@link Color} of highlights
     * @param highlights 
     */
    public void setHighlights(Color highlights) {
        this.highlights = highlights;
    }
    
    /**
     * @return Value of {@link #renderMode}
     */
    public RenderingMode getRenderMode() {
        return renderMode;
    }

    /**
     * Sets rendering mode
     * @param renderMode Rendering mode
     */
    public void setRenderMode(RenderingMode renderMode) {
        this.renderMode = renderMode;
    }

    protected void setPolygonMode(GL2 gl) {
        gl.glShadeModel(GL2.GL_SMOOTH);
        switch (getRenderMode()) {
            case WIREFRAME:
                gl.glPolygonMode(GL.GL_FRONT_AND_BACK, GL2.GL_LINE);
                break;
            case POINT_CLOUD:
                gl.glPolygonMode(GL.GL_FRONT_AND_BACK, GL2.GL_POINT);
                break;
            default:
            case TEXTURE:
            case SMOOTH:
                gl.glPolygonMode(GL.GL_FRONT_AND_BACK, GL2.GL_FILL);
                break;                
        }
    }
}
