package cz.fidentis.analyst.featurepoints;

import cz.fidentis.analyst.core.LoadedActionEvent;
import cz.fidentis.analyst.feature.FeaturePoint;
import cz.fidentis.analyst.feature.FeaturePointType;
import cz.fidentis.analyst.feature.RelationToMesh;
import cz.fidentis.analyst.scene.DrawableFeaturePoints;
import java.awt.Color;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.swing.AbstractAction;
import javax.swing.JCheckBox;
import javax.swing.JLabel;
import javax.swing.JPanel;

/**
 * @author Katerina Zarska
 */
public class FeaturePointListPanel extends JPanel {
    
    public static final String ACTION_COMMAND_FEATURE_POINT_HOVER_IN = "highlight hovered FP";
    public static final String ACTION_COMMAND_FEATURE_POINT_HOVER_OUT = "set default color of hovered FP";
    public static final String ACTION_COMMAND_FEATURE_POINT_SELECT = "highlight feature point with color";
    
    private List<JLabel> colorIndicators;
    private Map<RelationToMesh, JLabel> featurePointRelations;
    private List<JCheckBox> featurePointCheckBoxes;
    
    /**
     * Constructor.
     * 
     * @param action Action listener
     * @param featurePoints List of all feature points 
     */
    public void initComponents(ActionListener action, DrawableFeaturePoints featurePoints) {
        setLayout(new GridBagLayout());
        
        featurePointRelations = new HashMap<>(featurePoints.getFeaturePoints().size());
        featurePointCheckBoxes = new ArrayList<>(featurePoints.getFeaturePoints().size());
        colorIndicators = new ArrayList<>(featurePoints.getFeaturePoints().size());
        
        for (int i = 0; i < featurePoints.getFeaturePoints().size(); i++) {
            //final FeaturePoint featurePoint = featurePoints.get(i);
            //final FeaturePointType featurePointType = featurePoint.getFeaturePointType();
            //final RelationToMesh relation = featurePoint.getRelationToMesh();
            //addRow(i, action, featurePointType, relation);
            addRow(i, action, featurePoints);
        }
    }
    
    /**
     * (De)selects feature points
     * 
     * @param selected {@code true} if all feature point checkboxes are to be selected,
     *                 {@code false} otherwise
     */
    public void selectFeaturePoints(List<FeaturePoint> featurePoints, List<FeaturePointType> selectedFeaturePoints, boolean selected) {
        for (int i = 0; i < featurePoints.size(); i++) {
            if (selectedFeaturePoints.contains(featurePoints.get(i).getFeaturePointType())) {
                featurePointCheckBoxes.get(i).setSelected(selected);
            } else {
                featurePointCheckBoxes.get(i).setSelected(!selected);
            }
        }
    }
    
    /**
     * (De)selects given feature point
     * 
     * @param index Index of the feature point
     * @param selected {@code true} if a feature point checkbox is to be selected,
     *                 {@code false} otherwise
     */
    public void selectFeaturePoint(int index, boolean selected) {
        if (index >= 0 && index < featurePointCheckBoxes.size()) {
            featurePointCheckBoxes.get(index).setSelected(selected);
        }
    }
    
    /**
     * Refreshes the panel
     * @param action
     * @param featurePoints 
     */
    public void refreshPanel(ActionListener action, DrawableFeaturePoints featurePoints, List<FeaturePointType> selectedFeaturePoints) {
        removeAll();
        initComponents(action, featurePoints);
        revalidate();
        repaint();
        selectFeaturePoints(featurePoints.getFeaturePoints(), selectedFeaturePoints, true);
    }
    
    /**
     * Creates and returns action listener that can be connected with a low-level 
     * GUI element (e.g., a button). Action event of triggered low-level element is
     * redirected to the given {@code action}.
     * The listener may also carry additional data as a payload.
     * 
     * @param action Action listener
     * @param command Control panel command
     * @param data Payload data of the action listener
     * @return Action listener
     */
    protected final ActionListener createListener(ActionListener action, String command, Object data) {
        return new AbstractAction() {
            @Override
            public void actionPerformed(ActionEvent e) {
                action.actionPerformed(new LoadedActionEvent(
                        e.getSource(), 
                        ActionEvent.ACTION_PERFORMED, 
                        command,
                        data)
                ); 
            }  
        };
    }
    
    protected void addRow(int row, ActionListener action, DrawableFeaturePoints featurePoints) {//FeaturePointType featurePointType, RelationToMesh relation) {
        FeaturePoint featurePoint = featurePoints.getFeaturePoints().get(row);
        
        Color color = DrawableFeaturePoints.OFF_THE_MESH_COLOR;
        if (featurePoint.getRelationToMesh().getPositionType() == RelationToMesh.FpPositionType.ON_THE_MESH) {
            color = DrawableFeaturePoints.ON_THE_MESH_COLOR;
        } else if (featurePoint.getRelationToMesh().getPositionType() == RelationToMesh.FpPositionType.CLOSE_TO_MESH) {
            color = DrawableFeaturePoints.CLOSE_TO_MESH_COLOR;
        }

        GridBagConstraints c = new GridBagConstraints();  
        c.gridwidth = 1;
        c.gridx = 0;
        c.gridy = row;
        c.anchor = GridBagConstraints.LINE_START;
        c.fill = GridBagConstraints.NONE;
        
        GridBagConstraints c1 = new GridBagConstraints();
        c1.insets = new Insets(0, 10, 0, 10);
        c1.gridwidth = GridBagConstraints.REMAINDER;
        c1.gridx = 1;
        c1.gridy = row;
        c1.anchor = GridBagConstraints.LINE_END;
        c1.fill = GridBagConstraints.NONE;
        
        JLabel label = new JLabel("⚫");
        label.setForeground(color);
        add(label, c1);
        colorIndicators.add(label);
        
        JCheckBox checkBox = new JCheckBox();
        checkBox.setText(featurePoint.getFeaturePointType().getName());      
        add(checkBox, c);
        featurePointCheckBoxes.add(checkBox);
        
        checkBox.addActionListener(createListener(action, ACTION_COMMAND_FEATURE_POINT_SELECT, row));
        
        final int index = row;
        checkBox.addMouseListener(new MouseAdapter() { // highlight spheres on mouse hover
            @Override
            public void mouseEntered(MouseEvent e) {
                action.actionPerformed(new LoadedActionEvent(
                        e.getSource(),
                        ActionEvent.ACTION_PERFORMED,
                        ACTION_COMMAND_FEATURE_POINT_HOVER_IN,
                        index));
            }

            @Override
            public void mouseExited(MouseEvent e) {
                action.actionPerformed(new LoadedActionEvent(
                        e.getSource(),
                        ActionEvent.ACTION_PERFORMED,
                        ACTION_COMMAND_FEATURE_POINT_HOVER_OUT,
                        index));
            }
        });
                
//        GridBagConstraints c2 = new GridBagConstraints();
//        c2.gridwidth = GridBagConstraints.REMAINDER;
//        c2.gridx = 2;
//        c2.gridy = row;
//        c2.anchor = GridBagConstraints.LINE_END;
//        c2.fill = GridBagConstraints.NONE;
//        c2.insets = new Insets(0, 0, 0, 0);
//        
//        JLabel label2 = new JLabel();
//        label2.setText(featurePoint.getRelationToMesh().getPositionTypeToString());
//        label2.setVisible(true);
//        add(label2, c2);
//        featurePointRelations.put(featurePoint.getRelationToMesh(), label2);
    }
}
