package cz.fidentis.analyst.core;

import cz.fidentis.analyst.Project;
import cz.fidentis.analyst.batch.BatchAction;
import cz.fidentis.analyst.canvas.Canvas;
import cz.fidentis.analyst.canvas.toolbar.SceneToolboxFaceToFace;
import cz.fidentis.analyst.canvas.toolbar.SceneToolboxSingleFace;
import cz.fidentis.analyst.curvature.CurvatureAction;
import cz.fidentis.analyst.distance.DistanceAction;
import cz.fidentis.analyst.face.HumanFace;
import cz.fidentis.analyst.faceState.FaceStatePanel;
import cz.fidentis.analyst.project.Task;
import cz.fidentis.analyst.registration.RegistrationAction;
import cz.fidentis.analyst.scene.Camera;
import cz.fidentis.analyst.symmetry.ProfilesAction;
import cz.fidentis.analyst.symmetry.SymmetryAction;
import cz.fidentis.analyst.featurepoints.FeaturePointsAction;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import javax.swing.GroupLayout;
import javax.swing.JScrollPane;
import javax.swing.LayoutStyle;
import org.openide.windows.TopComponent;

/**
 * The non-singleton window/tab for the analysis of one, two or many to many faces
 *
 * @author Matej Kovar
 */
public class FaceTab extends TopComponent {
    
    private final Canvas canvas ;
    private final TopControlPanel controlPanel;
    private final JScrollPane scrollPane;
    
    private HumanFace face1 = null;
    private HumanFace face2 = null;
    private List<Path> faces = new ArrayList<>();
    
    private Project project;
    
    /**
     * Constructor for single face analysis and 1:1.
     * 
     * @param primary Primary face
     * @param secondary Secondary face
     * @param name Tab name
     * @param endTaskListener action endTaskListener
     * @param project Project
     */
    public FaceTab(HumanFace primary, HumanFace secondary, String name, 
            ActionListener endTaskListener, Project project) {
        
        canvas = new Canvas();
        this.project = project;
        
        face1 = primary;

        canvas.addPrimaryFace(primary);
        
        if (secondary == null) { // single face analysis
            canvas.getScene().setDefaultColors();
            canvas.addToolBox(new SceneToolboxSingleFace(canvas, 
                    (ActionEvent e) -> {
                        endTaskListener.actionPerformed(
                                new ActionEvent(this, 
                                ActionEvent.ACTION_PERFORMED, 
                                "FaceTab"));
                    }));
            
        } else { // 1:1
            canvas.addSecondaryFace(secondary);
            canvas.getScene().setDefaultColors();
            canvas.addToolBox(new SceneToolboxFaceToFace(canvas, 
                    (ActionEvent e) -> {
                        endTaskListener.actionPerformed(
                                new ActionEvent(this, 
                                ActionEvent.ACTION_PERFORMED, 
                                "FaceToFaceTab"));
                    }));
            face2 = secondary;
        }

        controlPanel = new TopControlPanel();
        scrollPane = new JScrollPane(controlPanel);
        
        setName(name);
        initComponents();
        
        // change the height so that it corresponds to the height of the OpenGL window
        canvas.addComponentListener(new ComponentAdapter() {
            @Override
            public void componentResized(ComponentEvent e) {
                scrollPane.setSize(ControlPanel.CONTROL_PANEL_WIDTH, canvas.getHeight());
            }
        });
        
        controlPanel.addChangeListener(e -> getCanvas().renderScene());
            
        if (secondary == null) { // single face analysis
            new CurvatureAction(getCanvas(), controlPanel);
            new SymmetryAction(getCanvas(), controlPanel);
            new ProfilesAction(getCanvas(), controlPanel);
            new FeaturePointsAction(getCanvas(), controlPanel);
            
            // Face State Panel for primary face
            FaceStatePanel facePanel = this.createFacePanelStateForFace(primary, project);
            controlPanel.addTab(facePanel.getName(), facePanel.getIcon(), facePanel);
            
        } else { // 1:1
            new RegistrationAction(canvas, controlPanel);
            new DistanceAction(canvas, controlPanel);
            new SymmetryAction(canvas, controlPanel);
            new ProfilesAction(canvas, controlPanel);
            
            // Face State Panel for primary face
            FaceStatePanel facePanel = this.createFacePanelStateForFace(primary, project);
            facePanel.setTypeOfIcon(FaceStatePanel.TypeOfIcon.MANY_TO_MANY_PRIMARY);
            controlPanel.addTab(facePanel.getName(), facePanel.getIcon(), facePanel);
            
            // Face State Panel for secondary face
            FaceStatePanel secondaryFacePanel = this.createFacePanelStateForFace(secondary, project);
            secondaryFacePanel.setTypeOfIcon(FaceStatePanel.TypeOfIcon.MANY_TO_MANY_SECONDARY);
            controlPanel.addTab(secondaryFacePanel.getName(), secondaryFacePanel.getIcon(), secondaryFacePanel);
        }
    }
    
    /**
     * Constructor for N:N
     * 
     * @param faces Paths to faces
     * @param name Tab name
     * @param endTaskListener action endTaskListener
     */
    public FaceTab(List<Path> faces, String name, ActionListener endTaskListener) {
        canvas = new Canvas();
        
        canvas.addToolBox(new SceneToolboxSingleFace(canvas,                     
                    (ActionEvent e) -> {
                        endTaskListener.actionPerformed(
                                new ActionEvent(this, 
                                ActionEvent.ACTION_PERFORMED, 
                                "FaceToFaceTab"));
                    }));

        controlPanel = new TopControlPanel();
        scrollPane = new JScrollPane(controlPanel);
        
        setName(name);
        initComponents();
        
        // change the height so that it corresponds to the height of the OpenGL window
        canvas.addComponentListener(new ComponentAdapter() {
            @Override
            public void componentResized(ComponentEvent e) {
                scrollPane.setSize(ControlPanel.CONTROL_PANEL_WIDTH, canvas.getHeight());
            }
        });
        
        controlPanel.addChangeListener(e -> getCanvas().renderScene());
        
        this.faces = faces;
        new BatchAction(canvas, controlPanel, faces);
    }
    /*
    public FaceTab() {
        this(null, null, null, null, null);
    }*/
    
    @Override
    public int getPersistenceType() {
        return TopComponent.PERSISTENCE_NEVER; // TO DO: change to .PERSISTENCE_ONLY_OPENED when we can re-create the ProjectTC
    }
    
    private void initComponents() {
        GroupLayout layout = new GroupLayout(this);
        this.setLayout(layout);
        layout.setHorizontalGroup(
                layout.createParallelGroup(GroupLayout.Alignment.LEADING)
                        .addGroup(layout.createSequentialGroup()
                                .addPreferredGap(LayoutStyle.ComponentPlacement.RELATED)
                                .addComponent(canvas, GroupLayout.DEFAULT_SIZE, 651, Short.MAX_VALUE)
                                .addPreferredGap(LayoutStyle.ComponentPlacement.RELATED)
//                                .addComponent(renderingToolBar, GroupLayout.PREFERRED_SIZE, RenderingToolBar.WIDTH, GroupLayout.PREFERRED_SIZE)
                                .addComponent(
                                        scrollPane,
                                        ControlPanel.CONTROL_PANEL_WIDTH, 
                                        ControlPanel.CONTROL_PANEL_WIDTH, 
                                        ControlPanel.CONTROL_PANEL_WIDTH
                                )
                        )
        );
        layout.setVerticalGroup(
                layout.createParallelGroup(GroupLayout.Alignment.LEADING)
                        .addGroup(layout.createSequentialGroup()
                                .addGroup(layout.createBaselineGroup(true, true)
                                        .addComponent(canvas)
//                                        .addComponent(renderingToolBar)
                                        .addComponent(scrollPane)
                                ))
        );
    }

    /**
     * Gets name of primary face
     * @return name of primary face, or null
     */
    public String getNameOfFace1() {
        
        if (face1 != null) {
            return face1.getShortName();
        }
        return null;
    }

    /**
     * Gets name of secondary face
     * @return name of secondary face, or null
     */
    public String getNameOfFace2() {
        if (face2 != null) {
            return face2.getShortName();
        }
        return null;
    }
    
    public HumanFace getFace1() {
        return face1;
    }
    
    public HumanFace getFace2() {
        return face2;
    }
    
    public List<Path> getFaces() {
        return faces;
    }
    
    /**
     * Checks whether this tab contains name of face
     * @param name String name of face
     * @return true if face with this name is in this tab
     */
    public boolean hasFace(String name) {
        return (name.equals(getNameOfFace1()) || name.equals(getNameOfFace2()));
    }

    public Canvas getCanvas() {
        return canvas;
    }
    
    /**
     * Sets camera to canvas
     * @param camera 
     */
    public void setCamera(Camera camera) {
        this.canvas.setCamera(camera);
    }

    /**
     * Create FaceStatePanel for face
     * @param face which will be shown on panel
     * @param project project where path to this face is stored
     * @return new created face state panel
     */
    private FaceStatePanel createFacePanelStateForFace(HumanFace face, Project project) {
        Path pathToFace = Paths.get(face.getPath());
        String faceName = face.getShortName();
        FaceStatePanel facePanel = new FaceStatePanel(project, faceName, pathToFace);        
        facePanel.loadFaceGeometryInfo(face);
        facePanel.showFaceState(faceName, pathToFace);
        //controlPanel.addTab(facePanel.getName(), facePanel.getIcon(), facePanel);
        return facePanel;
    }
    
    /**
     * Create task for this tab and serialize data
     * @throws IOException 
     */
    public void serializeTask() throws IOException {
        if (face1 != null) {
            Task task = new Task(face1, face2, canvas.getCamera());
        
            task.dumpToFile(project.getPathToProjectFile()
                        .substring(0, project.getPathToProjectFile().lastIndexOf(File.separatorChar)));
        }
        
    }

    @Override
    public int hashCode() {
        int hash = 5;
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final FaceTab other = (FaceTab) obj;
        if (!Objects.equals(this.getNameOfFace1(), other.getNameOfFace1())) {
            return false;
        }
        if (!Objects.equals(this.getNameOfFace2(), other.getNameOfFace2())) {
            return false;
        }
        return Objects.equals(faces, other.getFaces());
    }

    
}