package cz.fidentis.analyst.visitors.mesh;

import cz.fidentis.analyst.mesh.core.MeshFacet;
import cz.fidentis.analyst.mesh.core.MeshPoint;
import cz.fidentis.analyst.mesh.core.MeshTriangle;
import java.util.List;
import javax.vecmath.Vector3d;

/**
 * Calculates max curvatures of mesh vertices using the combination
 * of Gaussian curvature and mean curvature (based on Laplacian function).
 * This algorithm is roughly 4-times slower than the pure Gaussian curvature algorithm,
 * but is more precise.
 * 
 * @author Natalia Bebjakova
 * @author Radek Oslejsek
 */
public class MaxCurvature extends GaussianCurvature {
    
    protected double calculateCurvature(MeshFacet facet, List<MeshTriangle> triangles, int centerIndex) {
        double gaussianCurvature = super.calculateCurvature(facet, triangles, centerIndex);
        if (Double.isNaN(gaussianCurvature)) {
            return Double.NaN;
        }
        double meanCurvature = getMeanCurvature(facet, triangles, centerIndex);
        double meanCurvatureSqr = meanCurvature * meanCurvature;
        if (meanCurvatureSqr <= gaussianCurvature) { 
            return meanCurvature;
        }
        return meanCurvature + Math.sqrt(meanCurvatureSqr - gaussianCurvature);
    }
    
    /**
     * 
     * @param centerIndex center index
     * @return mean curvature
     */
    private double getMeanCurvature(MeshFacet facet, List<MeshTriangle> triangles, int centerIndex) {
        return calculateLaplacian(facet, triangles, centerIndex).length();
    }
    
    /**
     * Calculates Laplacian 
     * 
     * @param centerIndex centerIndex
     * @return laplacian 
     */
    private Vector3d calculateLaplacian(MeshFacet facet, List<MeshTriangle> triangles, int centerIndex) {
        List<Integer> neighbouringTriangles = facet.getCornerTable().getTriangleIndexesByVertexIndex(centerIndex);

        if (neighbouringTriangles.isEmpty()) {
            return new Vector3d();
        }
        
        double areaSum = 0;
        Vector3d pointSum = new Vector3d();
        List<Vector3d> voronoiPoints = facet.calculateVoronoiPoints();
        
        for (int i = 0; i < neighbouringTriangles.size(); i++) {
            MeshTriangle t = triangles.get(neighbouringTriangles.get(i));
            MeshTriangle tNext = triangles.get(neighbouringTriangles.get((i + 1) % neighbouringTriangles.size()));
            double area = 0;
            
            MeshPoint centerPoint = facet.getVertices().get(centerIndex);
            Vector3d voronoiPoint = voronoiPoints.get(neighbouringTriangles.get(i));

            Vector3d aux = new Vector3d();
            if (t.vertex1 == facet.getVertex(centerIndex)) {
                area = voronoiPoint.x;
                aux = new Vector3d(t.vertex2.getPosition());
            } else if (t.vertex2 == facet.getVertex(centerIndex)) {
                area = voronoiPoint.y;
                aux = new Vector3d(t.vertex3.getPosition());
            } else if (t.vertex3 == facet.getVertex(centerIndex)) {
                area = voronoiPoint.z;
                aux = new Vector3d(t.vertex1.getPosition());
            }
            aux.sub(centerPoint.getPosition());
            aux.scale((calculateTriangleAlfa(centerPoint, t) + calculateTriangleBeta(centerPoint, tNext)) / 2.0);
            pointSum.add(aux);
            areaSum += area;
        }
        pointSum.scale(1.0/areaSum);
        return pointSum;
    }
    
    /**
     * Calculates angle of the triangle according to center point
     * 
     * @param centerPoint center point
     * @param t triangle
     * @return angle
     */
    private double calculateTriangleAlfa(MeshPoint centerPoint, MeshTriangle t) {        
        double alfaCos = 0;
        
        if (t.vertex1 == centerPoint) {
            MeshPoint e1 = t.vertex1.subtractPosition(t.vertex3);
            e1 = e1.dividePosition(e1.abs());
            
            MeshPoint e2 = t.vertex2.subtractPosition(t.vertex3);
            e2 = e2.dividePosition(e2.abs());
            
            alfaCos = e1.dotProduct(e2);
        } else if (t.vertex2 == centerPoint) {
            MeshPoint e1 = t.vertex2.subtractPosition(t.vertex1);
            e1 = e1.dividePosition(e1.abs());
            
            MeshPoint e2 = t.vertex3.subtractPosition(t.vertex1);
            e2 = e2.dividePosition(e2.abs());
            
            alfaCos = e1.dotProduct(e2);
        } else if (t.vertex3 == centerPoint){
            MeshPoint e1 = t.vertex3.subtractPosition(t.vertex2);
            e1 = e1.dividePosition(e1.abs());
                
            MeshPoint e2 = t.vertex1.subtractPosition(t.vertex2);
            e2 = e2.dividePosition(e2.abs());
            
            alfaCos = e1.dotProduct(e2);
        }
        double alfa = Math.acos(alfaCos);
        return 1 / Math.tan(alfa);
    }
    
    /**
     * Calculates angle of the triangle according to center point
     * 
     * @param centerPoint center point
     * @param t triangle
     * @return angle
     */
    private double calculateTriangleBeta(MeshPoint centerPoint, MeshTriangle t) {
        double betaCos = 0;
            
        if (t.vertex1 == centerPoint) {
            MeshPoint e1 = t.vertex1.subtractPosition(t.vertex2);
            e1 = e1.dividePosition(e1.abs());
            
            MeshPoint e2 = t.vertex3.subtractPosition(t.vertex2);
            e2 = e2.dividePosition(e2.abs());
            
            betaCos = e1.dotProduct(e2);
        } else if (t.vertex2 == centerPoint) {
            MeshPoint e1 = t.vertex2.subtractPosition(t.vertex3);
            e1 = e1.dividePosition(e1.abs());
            
            MeshPoint e2 = t.vertex1.subtractPosition(t.vertex3);
            e2 = e2.dividePosition(e2.abs());
            
            betaCos = e1.dotProduct(e2);
        } else if (t.vertex3 == centerPoint){
            MeshPoint e1 = t.vertex3.subtractPosition(t.vertex1);
            e1 = e1.dividePosition(e1.abs());
                
            MeshPoint e2 = t.vertex2.subtractPosition(t.vertex1);
            e2 = e2.dividePosition(e2.abs());
            
            betaCos = e1.dotProduct(e2);
        }
            double beta = Math.acos(betaCos);
            return 1 / Math.tan(beta);
    }
}
