package cz.fidentis.analyst.visitors.mesh;

import cz.fidentis.analyst.mesh.MeshVisitor;
import cz.fidentis.analyst.mesh.core.MeshFacet;
import cz.fidentis.analyst.mesh.core.MeshTriangle;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Abstract class for algorithms calculating curvatures of mesh vertices.
 * 
 * @author Natalia Bebjakova
 * @author Radek Oslejsek
 */
public abstract class Curvature extends MeshVisitor {
    
    private final Map<MeshFacet, List<Double>> curvatures = new HashMap<>();
    
    /**
     * Returns curvatures for all inspected mesh facets. The order corresponds to
     * the order of vertices, i.e., i-th value represents the curvature of i-th mesh vertex.
     * 
     * @return curvatures of inspected mesh facets.
     */
    public Map<MeshFacet, List<Double>> getCurvatures() {
        return Collections.unmodifiableMap(curvatures);
    }
    
    @Override
    public void visitMeshFacet(final MeshFacet facet) {
        synchronized (this) {
            if (curvatures.containsKey(facet)) {
                return; // already visited facet
            }
            curvatures.put(facet, new ArrayList<>());
            if (!facet.hasVertexNormals()) {
                facet.calculateVertexNormals();
            }
            facet.calculateVoronoiPoints();
        }
        
        final List<MeshTriangle> triangles = facet.getTriangles();
        for (int i = 0; i < facet.getNumberOfVertices(); i++) {
            curvatures.get(facet).add(calculateCurvature(facet, triangles, i));
        }
    }
    
    protected abstract double calculateCurvature(MeshFacet facet, List<MeshTriangle> triangles, int centerIndex);
    
}
