package cz.fidentis.analyst.visitors.mesh;

import cz.fidentis.analyst.mesh.MeshVisitor;
import cz.fidentis.analyst.mesh.core.MeshFacet;
import cz.fidentis.analyst.mesh.core.MeshPoint;
import cz.fidentis.analyst.mesh.core.MeshPointImpl;
import java.util.List;
import javax.vecmath.Vector3d;

/**
 * Visitor that computes a 3D bounding box (cube).
 * <p>
 * This visitor is thread-safe. 
 * </p>
 * 
 * @author Radek Oslejsek
 */
public class BoundingBox extends MeshVisitor {

    private BBox bbox;
    
    @Override
    public synchronized void visitMeshFacet(MeshFacet facet) {
        if (bbox == null) {
            bbox = new BBox(facet.getVertices());
        } else {
            bbox.compute(facet.getVertices());
        }
    }
    
    /**
     * Returns computed bounding box.
     * 
     * @return Bounding box or {@code null}
     */
    public BBox getBoundingBox() {
        return bbox;
    }
    
    /**
     * 3D bounding box (cube) of {@link MeshPoint}s.
     * 
     * @author Natalia Bebjakova
    */
    public class BBox {
    
        private MeshPoint maxPoint;
        private MeshPoint minPoint;

        /** 
         * Creates bounding box from given mesh points.
         * 
         * @param points List of mesh points, must not be null or empty
         * @throws IllegalArgumentException if the @code{points} param is null or empty
         */
        public BBox(List<MeshPoint> points) {
            if (points == null || points.isEmpty()) {
                throw new IllegalArgumentException("points");
            }

            minPoint = new MeshPointImpl(new Vector3d(Double.POSITIVE_INFINITY, Double.POSITIVE_INFINITY, Double.POSITIVE_INFINITY), null, null);
            maxPoint = new MeshPointImpl(new Vector3d(Double.NEGATIVE_INFINITY,Double.NEGATIVE_INFINITY,Double.NEGATIVE_INFINITY), null, null);

            compute(points);
        }

        /** 
         * Creates bounding box from given mesh points, possibly extends the existing BB.
         * 
         * @param points List of mesh points, must not be null or empty
         */
        public void compute(List<MeshPoint> points) {
            if (points == null || points.isEmpty()) {
                return;
            }

            for (int i = 0; i < points.size(); i++) {
                MeshPoint point = points.get(i);

                minPoint.getPosition().x = Math.min(minPoint.getPosition().x, point.getPosition().x);
                minPoint.getPosition().y = Math.min(minPoint.getPosition().y, point.getPosition().y);
                minPoint.getPosition().z = Math.min(minPoint.getPosition().z, point.getPosition().z);

                maxPoint.getPosition().x = Math.max(maxPoint.getPosition().x, point.getPosition().x);
                maxPoint.getPosition().y = Math.max(maxPoint.getPosition().y, point.getPosition().y);
                maxPoint.getPosition().z = Math.max(maxPoint.getPosition().z, point.getPosition().z);                    
            }
        }        

        /**
         * Return the upper-bound corner of the bounding cube
         * @return max point of the bounding box
         */
        public MeshPoint getMaxPoint() {
            return maxPoint;
        }

        /**
         * Return centroid of the bounding cube.
         * @return middle point of the bounding box
         */
        public MeshPoint getMidPoint() {
            return (minPoint.addPosition(maxPoint)).multiplyPosition(0.5);
        }

        /**
         * Return the lower-bound corner of the bounding cube
         * @return min point of the bounding box
         */
        public MeshPoint getMinPoint() {
            return minPoint;
        }

        /**
         * Return volume diagonal of the bounding box.
         * @return maximal diagonal of bounding box
         */
        public double getMaxDiag() {
            return maxPoint.subtractPosition(minPoint).abs();
        }

        /**
         * Returns description of BoundignBox.
         * 
         * @return String representation of the bounding box
         */
        @Override
        public String toString() {
            String str = "BoundingBox: ";
            str += System.lineSeparator();
            str += "\t" + "- min point : " + this.minPoint + System.lineSeparator();
            str += "\t" + "- max point : " + this.maxPoint + System.lineSeparator();
            return str;
        }      

    }
}
