package cz.fidentis.analyst.tests;

import cz.fidentis.analyst.BatchProcessor;
import cz.fidentis.analyst.face.HumanFace;
import cz.fidentis.analyst.face.HumanFaceFactory;
import cz.fidentis.analyst.icp.IcpTransformation;
import cz.fidentis.analyst.icp.IcpTransformer;
import cz.fidentis.analyst.icp.RandomStrategy;
import cz.fidentis.analyst.icp.UndersamplingStrategy;
import cz.fidentis.analyst.kdtree.KdTree;
import cz.fidentis.analyst.mesh.core.MeshFacet;
import cz.fidentis.analyst.mesh.io.MeshObjExporter;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * A class for testing the efficiency of batch processing algorithms.
 * 
 * @author Radek Oslejsek
 */
public class BatchTests {
    
    private static final String dataDir = "../../analyst-data/multi-scan-models-anonymized-fixed/";
    private static final String primaryFacePath = dataDir + "/average-boy-17-20/average_boy_17-20.obj";
    //private static final String primaryFacePath = dataDir + "/average-girl-17-20/average_girl_17-20.obj";
    //private static final String primaryFacePath = "../../analyst-data/basic-models/02.obj";
    private static final String templateFacePath = dataDir + "/template.obj";
    
    /**
     * Main method 
     * @param args Input arguments 
     * @throws IOException on IO error
     */
    public static void main(String[] args) throws IOException, ClassNotFoundException, Exception {
        BatchProcessor batch = new BatchProcessor();
        long startTime, endTime;
        
        List<String> faceIDs;
        String faceId = HumanFaceFactory.instance().loadFace(new File(primaryFacePath));
        HumanFace primaryFace = HumanFaceFactory.instance().getFace(faceId);
        KdTree kdTree = primaryFace.computeKdTree(false);
        
        //////////////////////////////////////////////
        System.out.println();
        System.out.println("Loading files:");
        startTime = System.currentTimeMillis();
        faceIDs = readFiles(dataDir);
        endTime = System.currentTimeMillis();
        printTimes(startTime, endTime, faceIDs.size());
        System.out.println("Number of files: " + faceIDs.size());
        System.out.println("File cache stats: " + HumanFaceFactory.instance());
        
        //////////////////////////////////////////////
        //UndersamplingStrategy strategy = new NoUndersampling();
        UndersamplingStrategy strategy = new RandomStrategy(0.5);
        System.out.println();
        System.out.println("ICP registration with " + strategy + ":");
        startTime = System.currentTimeMillis();
        batch.registerToPrimaryFace(primaryFacePath, faceIDs, 10, 0.05, strategy);
        endTime = System.currentTimeMillis();
        printTimes(startTime, endTime, faceIDs.size());
        System.out.printf("%.2f \t\t Avg number of iterations", batch.getAvgNumIcpIterations());
        System.out.println();
        
        //////////////////////////////////////////////
        System.out.println();
        System.out.println("Computation of averaged template face:");
        startTime = System.currentTimeMillis();
        batch.computeTemplateFace(primaryFacePath, faceIDs);
        endTime = System.currentTimeMillis();
        printTimes(startTime, endTime, faceIDs.size());
        System.out.println("Writing average template to " + templateFacePath + "...");
        MeshObjExporter exp = new MeshObjExporter(batch.getAvgTemplateFace().getMeshModel());
        exp.exportModelToObj(new File(templateFacePath));
    }
    
    private static List<String> readFiles(String dir) throws IOException {
        List<String> faces = new ArrayList<>();
        for (File subdir : (new File(dir)).listFiles(File::isDirectory)) {
            if (subdir.getName().matches("^\\d\\d$")) {
                for (File file: subdir.listFiles(File::isFile)) {
                    if (file.getName().endsWith(".obj")) {
                        //System.out.print(file.getName() + " ");
                        //if (file.getName().endsWith("00002_01_ECA.obj")) { // FOT DEBUGING
                            String faceId = HumanFaceFactory.instance().loadFace(file);
                            faces.add(faceId);
                        //}
                    }
                }
            }
        }
        //System.out.println();
        return faces;
    }
    
    private static void printTimes(long startTime, long endTime, int numFaces) {
        double timeDist = endTime - startTime;
        double avgTime = (timeDist/numFaces)/1000.0;
        System.out.printf("%.2f sec \t Overall time", timeDist/1000.0);
        System.out.println();
        System.out.printf("%.2f sec \t Average time", avgTime);
        System.out.println();
        System.out.printf("%.2f min \t Estimated time for 500 faces", (avgTime*500)/60);
        System.out.println();
    }
}
