package cz.fidentis.analyst.visitors.mesh;

import cz.fidentis.analyst.mesh.MeshVisitor;
import cz.fidentis.analyst.mesh.core.MeshFacet;
import cz.fidentis.analyst.mesh.core.MeshPoint;
import cz.fidentis.analyst.visitors.DistanceWithNearestPoints;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.vecmath.Vector3d;

/**
 * This visitor finds the minimal distance between the given 3D point and mesh facets.
 * All closest mesh facets and all their closest vertices are returned.
 * <p>
 * This visitor is thread-safe.
 * </p>
 *
 * @author Matej Lukes
 * @author Radek Oslejsek
 */
public class MeshDistanceToVertices extends MeshVisitor implements DistanceWithNearestPoints {
    
    private double distance = Double.POSITIVE_INFINITY;
    private final Vector3d point3d;
    private Map<MeshFacet, List<Vector3d>> nearestPoints = new HashMap<>();
    
    /**
     * @param point A 3D point from which distance is computed. Must not be {@code null}
     * @throws IllegalArgumentException if some parameter is wrong
     */
    public MeshDistanceToVertices(Vector3d point) {
        if (point == null) {
            throw new IllegalArgumentException("point");
        }
        this.point3d = point;
    }

    @Override
    public Map<MeshFacet, List<Vector3d>> getNearestPoints() {
        return Collections.unmodifiableMap(nearestPoints);
    }

    @Override
    public double getDistance() {
        return distance;
    }

    @Override
    public void visitMeshFacet(MeshFacet facet) {
        List<MeshPoint> vertices = facet.getVertices();
        for (int i = 0; i < vertices.size(); i++) {
            Vector3d pointOnSurface = vertices.get(i).getPosition();
            Vector3d aux = new Vector3d(pointOnSurface);
            aux.sub(point3d);
            
            double dist = aux.length();
            
            synchronized (this) {
                if (dist > distance) {
                    continue;
                }
                if (dist < distance) {
                    distance = dist;
                    nearestPoints.clear();                    
                }
                nearestPoints.putIfAbsent(facet, new ArrayList<>());
                nearestPoints.get(facet).add(pointOnSurface);
            }
        }
    }
}    
