package cz.fidentis.analyst.face;

import java.io.IOException;

/**
 * This cache preserves the first X faces (privileged faces) always in the memory. 
 * Faces that are loaded when there is not enough free memory are always dumped
 * to file and only one of them is stored in the memory when used.
 * 
 * @author Radek Oslejsek
 */
public class HumanFacePrivilegedCache extends HumanFaceCache {
    
    private static HumanFacePrivilegedCache factory;
    
    /**
     * The only one dumped face currently in the memory
     */
    private String recoveredFace;
    
    /**
     * Private constructor. Use {@link instance} instead to get and access the instance.
     */
    protected HumanFacePrivilegedCache() {
    }
    
    /**
     * Returns the factory singleton instance.
     * 
     * @return the factory singleton instance
     */
    public static HumanFacePrivilegedCache instance() {
        if (factory == null) {
            factory = new HumanFacePrivilegedCache();
        }
        return factory;
    }

    @Override
    protected String selectFaceForDump() {
        if (this.recoveredFace != null) {
            return this.recoveredFace;
        } else {
            // This should not happen. But return any face from the memory, anyway.
            for (String faceId: getInMemoryFaces().keySet()) {
                return faceId; 
            }
        }
        return null;
    }
    
    @Override
    public void removeFace(String faceId) {
        if (faceId.equals(this.recoveredFace)) {
            this.recoveredFace = null;
        }
        super.removeFace(faceId);
    }
    
    @Override
    protected HumanFace recoverFace(String faceId) throws IOException, ClassNotFoundException {
        HumanFace face = super.recoverFace(faceId);
        if (face != null) {
            this.recoveredFace = face.getId();
        }
        return face;
    }
}
