package cz.fidentis.analyst.face;

import com.google.common.eventbus.EventBus;
import cz.fidentis.analyst.mesh.core.MeshFacet;
import cz.fidentis.analyst.mesh.core.MeshModel;
import cz.fidentis.analyst.mesh.events.MeshEvent;
import cz.fidentis.analyst.mesh.events.MeshListener;
import cz.fidentis.analyst.mesh.io.MeshObjLoader;
import cz.fidentis.analyst.symmetry.Plane;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.Objects;

/**
 * This class encapsulates data for a 3D scan of a single human face.
 * <p>
 * Changes in the human face and its data structures (e.g., mesh model, etc.)
 * can be monitored by listeners. Listeners have to implement the 
 * {@link cz.fidentis.analyst.face.HumanFaceListener} interface and they have to be
 * registered using the {@link cz.fidentis.analyst.face.HumanFace#registerListener} method.
 * Then they are informed about changes in the human automatically via methods 
 * prescribed by the interface.
 * </p>
 * <p>
 * Events fired by the class:
 * <ul>
 * <li>{@link cz.fidentis.analyst.mesh.events.MeshEvent} or its sub-type if 
 * the mesh model changes (see {@link cz.fidentis.analyst.mesh.core.MeshModel}
 * documentation for monitored changes).</li>
 * </ul>
 * </p>
 * 
 * @author Radek Oslejsek
 */
public class HumanFace implements MeshListener, Serializable {
    
    private final MeshModel meshModel;

    private Plane symmetryPlane;

    private MeshFacet cuttingPlane;
    
    private final transient EventBus eventBus = new EventBus();
    
    private final String id;
    
    /**
     * Reads a 3D human face from the given OBJ file.
     * Use {@link restoreFromFile} to restore the human face from a dump file.
     * 
     * @param file OBJ file
     * @throws IOException on I/O failure
     */
    public HumanFace(File file) throws IOException {
        meshModel = MeshObjLoader.read(new FileInputStream(file));
        meshModel.simplifyModel();
        meshModel.registerListener(this);
        this.id = file.getCanonicalPath();
    }
    
    /**
     * Returns the triangular mesh model of the human face.
     * 
     * @return the triangular mesh model of the human face
     */
    public MeshModel getMeshModel() {
        return meshModel;
    }

    /**
     * Registers listeners (objects concerned in the human face changes) to receive events.
     * If listener is {@code null}, no exception is thrown and no action is taken.
     * 
     * @param listener Listener concerned in the human face changes.
     */
    public void registerListener(MeshListener listener) {
        eventBus.register(listener);
    }
    
    /**
     * Unregisters listeners from receiving events.
     * 
     * @param listener Registered listener
     */
    public void unregisterListener(MeshListener listener) {
        eventBus.unregister(listener);
    }

    /**
     * Captures events fired by {@link cz.fidentis.analyst.mesh.core.MeshModel} and
     * redirects them to our listeners.
     * 
     * @param event A fired event.
     */
    @Override
    public void meshEvent(MeshEvent event) {
        eventBus.post(event);
    }

    /**
     *
     * @param plane The new symmetry plane
     */
    public void setSymmetryPlane(Plane plane) {
        symmetryPlane = plane;
    }

    /**
     *
     * @return The face's symmetry plane
     */
    public Plane getSymmetryPlane() {
        return symmetryPlane;
    }

    /**
     *
     * @param facet MeshFacet of the new cutting plane
     */
    public void setCuttingPlane(MeshFacet facet) {
        cuttingPlane = facet;
    }

    /**
     *
     * @return The face's cutting plane
     */
    public MeshFacet getCuttingPlane() {
        return cuttingPlane;
    }
    
    /**
     * Return unique ID of the face.
     * 
     * @return unique ID of the face. 
     */
    public String getId() {
        return this.id;
    }

    /**
     * Creates serialized dump of the human face. Event buses are not stored.
     * Therefore, listeners have to re-register again after recovery.
     * 
     * @return Dump file
     * @throws IOException on error in creating the dump file
     */
    public File dumpToFile() throws IOException {
        File tempFile = File.createTempFile(this.getClass().getSimpleName(), ".ser");
        tempFile.deleteOnExit();
        try (ObjectOutputStream fos = new ObjectOutputStream(new FileOutputStream(tempFile))) {
            fos.writeObject(this);
        }
        return tempFile;
    }
    
    /**
     * Restores human face from a dump file.
     * 
     * @param dumpFile The file
     * @return Human face
     * @throws IOException on error in reading the dump file
     * @throws java.lang.ClassNotFoundException on error when instantiating the human face
     */
    public static HumanFace restoreFromFile(File dumpFile) throws IOException, ClassNotFoundException {
        try (ObjectInputStream fos = new ObjectInputStream(new FileInputStream(dumpFile))) {
            return (HumanFace) fos.readObject();
        }
    }

    @Override
    public int hashCode() {
        int hash = 7;
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final HumanFace other = (HumanFace) obj;
        if (!Objects.equals(this.id, other.id)) {
            return false;
        }
        return true;
    }
    
    
}
