package cz.fidentis.analyst;

import cz.fidentis.analyst.face.HumanFace;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * This class encapsulates data for a "comparison project", i.e., a project 
 * enabling analysts to compare and investigate human faces.
 *
 * @author Matej Kovar
 */
public class Project {
    
    private HumanFace primaryFace;
    private List<HumanFace> secondaryFaces = new ArrayList<>();
    
    /**
     * Returns primary face
     * 
     * @return HumanFace primary face 
     */
    public HumanFace getPrimaryFace() {
        return primaryFace;
    }

    /**
     * Sets new primary face
     * 
     * @param primaryFace which will be new primary face
     * @throws IllegalArgumentException when argument primaryFace is null
     */
    public void setPrimaryFace(HumanFace primaryFace) {
        if (primaryFace == null) {
            throw new IllegalArgumentException("Primary face is null");
        }
        this.primaryFace = primaryFace;
    }
    
    /**
     * Removes primary face and sets primaryFace attribute to null
     */
    public void removePrimaryFace() {
        this.primaryFace = null;
    }

    /**
     * Returns list of HumanFace secondary faces
     * 
     * @return list of secondary faces
     */
    public List<HumanFace> getSecondaryFaces() {
        return Collections.unmodifiableList(secondaryFaces);
    }

    /**
     * Sets new list of secondary faces
     * 
     * @param secondaryFaces list of HumanFace which will be new list of 
     * secondary faces
     * @throws IllegalArgumentException if one of faces from argument is null
     */
    public void setSecondaryFaces(List<HumanFace> secondaryFaces) {
        this.secondaryFaces.clear();
        for (int i = 0; i < secondaryFaces.size(); i++) {
            if (secondaryFaces.get(i) == null) {
                throw new IllegalArgumentException("One of faces is null");
            }
        }
        this.secondaryFaces.addAll(secondaryFaces);
    }
    
    /**
     * Adds new face to secondaryFaces
     * 
     * @param face HumanFace which will be added to list of secondary faces
     * @throws IllegalArgumentException when argument face is null
     */
    public void addSecondaryFace(HumanFace face) {
        if (face == null) {
            throw new IllegalArgumentException("Face is null");
        }
        this.secondaryFaces.add(face);
    }
    
    /**
     * Removes HumanFace from secondaryFaces
     * 
     * @param face HumanFace which will be removed from secondaryFaces
     * @throws IllegalArgumentException when argument face is null
     */
    public void removeSecondaryFace(HumanFace face) {
        if (face == null) {
            throw new IllegalArgumentException("Face is null");
        }
        
        for (int i = 0; i < secondaryFaces.size(); i++) {
            if (secondaryFaces.get(i).equals(face)) {
                secondaryFaces.remove(i);
            }
        } 
    }
    
    /**
     * Removes faces which are sent to this function by list of HumanFace
     * from secondaryFaces
     * 
     * @param faces List of HumanFace faces which should be removed from
     * secondaryFaces
     */
    public void removeSelected(List<HumanFace> faces) {
        for (int i = 0; i < faces.size(); i++) {
            this.removeSecondaryFace(faces.get(i));
        }
    }
    
    /* TODO implement comparable and create comparators for filtering
    secondaryFaces */
}
