package cz.fidentis.analyst.face;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Singleton flyweight factory that creates and caches human faces. Faces are
 * stored in the memory until there is enough space in the Java heap. Then they 
 * are cached on disk automatically. The dumping strategy can be switched in run-time.
 * <p>
 * Currently, listeners registered to {@code HumanFace} and {@code KdTree} 
 * are neither dumped nor recovered!
 * </p>
 * 
 * @author Radek Oslejsek
 */
public class HumanFaceFactory {
    
    /**
     * Dumping strategies.
     * @author Radek Oslejsek
     */
    public enum Strategy {
        LRU, // least recently used faces are dumped first
        MRU  // most recently used faces are dumped first
    }
    
    /**
     * Keep at least this portion of the Java heap memory free
     */
    public static final double MIN_FREE_MEMORY = 0.05; // 5%
    
    /**
     * Human faces currently being stored on disk.
     * <ul>
     * <li>Key = Face ID<li>
     * <li>Value = Dump file<li>
     * </ul>
     */
    private final Map<String, File> dumpedFaces = new HashMap<>();
    
    /**
     * Human faces currently being allocated in the memory.
     * <ul>
     * <li>Key = Last access time (milliseconds)<li>
     * <li>Value = Human face<li>
     * </ul>
     */
    private final SortedMap<Long, HumanFace> inMemoryFaces = new TreeMap<>();
    
    /**
     * The usage "table"
     * <ul>
     * <li>Key = Face ID<li>
     * <li>Value = Last access time (milliseconds)<li>
     * </ul>
     */
    private final Map<String, Long> usage = new HashMap<>();
    
    /**
     * The singleton instance
     */
    private static HumanFaceFactory factory;
    
    /**
     * Dumping strategy.
     */
    private Strategy strategy = Strategy.LRU;
    
    
    /**
     * Private constructor. Use {@code instance()} method instead to get the instance.
     */
    protected HumanFaceFactory() {
    }
    
    /**
     * Returns the factory singleton instance.
     * 
     * @return the factory singleton instance
     */
    public static HumanFaceFactory instance() {
        if (factory == null) {
            factory = new HumanFaceFactory();
        }
        return factory;
    }

    /**
     * Changes the dumping strategy
     * @param strategy Dumping strategy. Must not be {@code null}
     * @throws IllegalArgumentException if the strategy is missing
     */
    public void setStrategy(Strategy strategy) {
        if (strategy == null) {
            throw new IllegalArgumentException("strategy");
        }
        this.strategy = strategy;
    }
    
    /**
     * Returns current dumping strategy.
     * @return current dumping strategy
     */
    public Strategy getStrategy() {
        return this.strategy;
    }
    
    /**
     * Loads new face. If the face is already loaded, then the ID of existing instance
     * is returned. To access the human face instance, use {@link getFace}.
     * 
     * @param file OBJ file with human face.
     * @return ID of the human face or {@code null}
     */
    public String loadFace(File file) {
        // Create face ID:
        String faceId;
        try {
            faceId = file.getCanonicalPath();
            //faceId = Long.toHexString(Double.doubleToLongBits(Math.random()));
        } catch (IOException ex) {
            Logger.getLogger(HumanFaceFactory.class.getName()).log(Level.SEVERE, null, ex);
            return null;
        }
        
        // Existing face
        HumanFace face = getFace(faceId);
        if (face != null) {
            return faceId;
        }
        
        // New face -- free the memory and load human face simultaneously:
        try {
            ExecutorService executor = Executors.newFixedThreadPool(Runtime.getRuntime().availableProcessors());
            final Future<HumanFace> result = executor.submit(() -> new HumanFace(file)); // read from file
            executor.submit(() -> checkMemAndDump()); // free the memory, if necessary
            executor.shutdown();
            while (!executor.isTerminated()){}
            face = result.get();
        } catch (InterruptedException|ExecutionException ex) {
            Logger.getLogger(HumanFaceFactory.class.getName()).log(Level.SEVERE, null, ex);
            return null;
        }
        
        // Update data structures:
        long time = System.currentTimeMillis();
        inMemoryFaces.put(time, face);
        usage.put(faceId, time);
        return faceId;
    }
    
    /**
     * Returns a human face. Recovers the face from the disk if necessary.
     * 
     * @param faceId ID of the face
     * @return Human face or {@code null}
     */
    public HumanFace getFace(String faceId) {
        if (updateAccessTime(faceId)) { // in memory face
            return inMemoryFaces.get(usage.get(faceId));
        }
        
        File dumpFile = dumpedFaces.get(faceId);
        if (dumpFile == null) { // unknown face
            return null;
        }
         
        // Free memory and recover human face from dump file silultanously:
        HumanFace face;
        try {
            ExecutorService executor = Executors.newFixedThreadPool(Runtime.getRuntime().availableProcessors());
            final Future<HumanFace> result = executor.submit(() -> HumanFace.restoreFromFile(dumpFile)); // recover face from disk
            executor.submit(() -> checkMemAndDump()); // free the memory, if necessary
            executor.shutdown();
            while (!executor.isTerminated()){}
            face = result.get();
        } catch (InterruptedException|ExecutionException ex) {
            Logger.getLogger(HumanFaceFactory.class.getName()).log(Level.SEVERE, null, ex);
            return null;
        }
        
        // Update data structures:
        long time = System.currentTimeMillis();
        dumpedFaces.remove(faceId);
        inMemoryFaces.put(time, face);
        usage.put(faceId, time);
        return face;
    }
    
    @Override
    public String toString() {
        StringBuilder ret = new StringBuilder();
        ret.append(formatSize(presumableFreeMemory())).
                append(" Java heap memory available (out of ").
                append(formatSize(Runtime.getRuntime().maxMemory()));
        ret.append("). In memory: ").
                append(this.inMemoryFaces.size()).
                append(", dumped: ").
                append(this.dumpedFaces.size());
        return ret.toString();
    }
    
    /**
     * Updates last access time of the face allocated in the memory. 
     * Returns {@code false} if the face is not in the memory.
     * 
     * @param faceId Face to be updated
     * @return {@code false} if the face is not in the memory.
     */
    protected boolean updateAccessTime(String faceId) {
        Long oldTime = usage.get(faceId);
        if (oldTime == null) {
            return false;
        }
        
        HumanFace face = inMemoryFaces.get(oldTime);
        
        long newTime = System.currentTimeMillis();
        inMemoryFaces.remove(oldTime);
        inMemoryFaces.put(newTime, face);
        usage.put(faceId, newTime);
        return true;
    }
    
    /**
     * Checks and releases the heap, if necessary.
     * 
     * @return true if some existing face has been dumped to free the memory.
     * @throws IOException on I/O error
     */
    protected boolean checkMemAndDump() throws IOException {
        double ratio = (double) presumableFreeMemory() / Runtime.getRuntime().maxMemory();
        if (ratio > MIN_FREE_MEMORY) {
            return false;
        }
        
        Long time = null;
        switch (strategy) {
            case MRU:
                time = inMemoryFaces.lastKey();
                break;
            default:
            case LRU:
                time = inMemoryFaces.firstKey();
                break;
        }
        
        HumanFace faceToDump = this.inMemoryFaces.remove(time);
        this.usage.remove(faceToDump.getId());
        dumpedFaces.put(faceToDump.getId(), faceToDump.dumpToFile());
        return true;
    }
        
    protected static String formatSize(long v) {
        if (v < 1024) {
            return v + " B";
        }
        int z = (63 - Long.numberOfLeadingZeros(v)) / 10;
        return String.format("%.1f %sB", (double)v / (1L << (z*10)), " KMGTPE".charAt(z));
    }
    
    /**
     * https://stackoverflow.com/questions/12807797/java-get-available-memory
     * @return 
     */
    protected long presumableFreeMemory() {
        long allocatedMemory = (Runtime.getRuntime().totalMemory()-Runtime.getRuntime().freeMemory());
        return Runtime.getRuntime().maxMemory() - allocatedMemory;
    }
}
